 MODULE CloudFractionModel
  USE wv_saturation, Only :findsp
  USE options, Only :iccon
  USE Constants, Only :r4,r8,cp,gasr  
  IMPLICIT NONE
  PRIVATE
  !INTEGER, PARAMETER, PUBLIC  :: r4 = SELECTED_REAL_KIND(6)  ! Kind for 32-bits Real Numbers
  !INTEGER, PARAMETER, PUBLIC  :: r8 = SELECTED_REAL_KIND(15) ! Kind for 64-bits Real Numbers
  !REAL (KIND=r8), PARAMETER   :: cp    =                  1004.6_r8! specific heat of air           (j/kg/k)
  !REAL (KIND=r8), PARAMETER   :: gasr  =                  287.05_r8! gas constant of dry air        (j/kg/k)
  !CHARACTER(LEN=*),PARAMETER  :: iccon='ARA'
  
  
  
  PUBLIC :: cldgn4

  CONTAINS
  
  !   This version is intended for, and was tested with, the new
  !shortwave radiation codes: clirad and ukmet 
  !
  SUBROUTINE cldgn4 (prsl  , &!REAL(KIND=r8),INTENT(in   ) :: prsl  (ncols,kMax)   ==> pressure   (Pa)
                     prsi  , &!REAL(KIND=r8),INTENT(in   ) :: prsi  (ncols,kMax+1) ==> interface layer pressure   (Pa)
                     shmt  , &!REAL(KIND=r8),INTENT(in   ) :: grh   (ncols,kmax)   ==> specific humidity  (kg/kg)
                     gice  , &!REAL(KIND=r8),INTENT(in   ) :: grh   (ncols,kmax)   ==> ice concentration  (kg/kg)
                     gliq  , &!REAL(KIND=r8),INTENT(in   ) :: grh   (ncols,kmax)   ==> liquid concentration  (kg/kg)
                     grh   , &!REAL(KIND=r8),INTENT(in   ) :: grh   (ncols,kmax)   ==> relative humidity  (0-1)
                     omg   , &!REAL(KIND=r8),INTENT(in   ) :: omg   (ncols,kmax)   ==> vertical velocity  (cb/sec)
                     gtmp_in, &!REAL(KIND=r8),INTENT(in   ) :: gtmp  (ncols,kmax)   ==> layer temperature (k)
                     ad_omg, &
                     ad_tmp, & 
                     ad_grh, & 
                     css   , &!REAL(KIND=r8),INTENT(inout) :: css   (ncols,kmax)   ==> supersatuation cloud cover fraction
                     ccu   , &!REAL(KIND=r8),INTENT(inout) :: ccu   (ncols,kmax)   ==> convective cloud cover fraction
                     cdin  , &!REAL(KIND=r8),INTENT(inout) :: cdin  (ncols,kmax)   ==> Inversion clouds
                     cstc  , &!REAL(KIND=r8),INTENT(inout) :: cstc  (ncols,kmax)   ==> Saturation clouds
                     ccon  , &!REAL(KIND=r8),INTENT(inout) :: ccon  (ncols,kmax)   ==> Convection clouds 
                     cson  , &!REAL(KIND=r8),INTENT(inout) :: cson  (ncols,kmax)   ==> Shallow convective clouds
                     convcin , &!REAL(KIND=r8),INTENT(in   ) :: convc (ncols)        ==> convective cloud cover in 3 hr. avrage
                     convtin , &!REAL(KIND=r8),INTENT(in   ) :: convt (ncols)        ==> ncols convective cloud top  (sigma layer)
                     convbin , &!REAL(KIND=r8),INTENT(in   ) :: convb (ncols)        ==> ncols convective cloud base (sigma layer)
                     lcnvl , &!INTEGER      ,INTENT(in   ) :: lcnvl       ==>  the lowest layer index where non-convective clouds can occur
                     lthncl, &!INTEGER      ,INTENT(in   ) :: lthncl      ==>  Minimum depth in mb of non-zero low level cloud constant
                     ncols , &!INTEGER      ,INTENT(IN   ) :: ncols       ==>  Number of colum
                     kmax  , &!INTEGER      ,INTENT(in   ) :: kmax        ==>  Number of layer
                     nls     )!INTEGER      ,INTENT(in   ) :: nls         ==>  Number of layers in the stratosphere.

    !==========================================================================
    ! cldgen :perform clouds generation scheme based on j. slingo
    !         (1984 ecmwf workshop); the scheme generates 4 type of clouds
    !         of convective, high, middle and low clouds.
    !==========================================================================
    ! parameters and input variables:
    !        covlp = 'maxi'         maximum overlap of convective cloud
    !                               or thick low cloud used in ir subr.
    !              = 'rand'         random  overlap of convective cloud
    !                               or thick low cloud used in ir subr.
    !        date  =  julian day of model forecast date
    !        grh   =  relative humidity  (0-1)
    !        omg   =  vertical velocity  (cb/sec)
    !        prsl  =  pressure   (Pa)
    !        sigmid   =  sigma coordinate at middle of layer
    !        gtmp  =  layer temperature (k)
    !     output variables:
    !        css   =  ncols*kmax supersatuation cloud cover fraction
    !        ccu   =  ncols*kmax convective cloud cover fraction
    !---------------------------------------------------------------------
    ! values from subr-gwater
    !       convc  =  ncols convective cloud cover in 3 hr. avrage
    !       convt  =  ncols convective cloud top  (sigma layer)
    !       convb  =  ncols convective cloud base (sigma layer)
    !==========================================================================
    !
    !   ncols.....Number of grid points on a gaussian latitude circle
    !   kmax......Number of grid points at vertical
    !   nls..... .Number of layers in the stratosphere.
    !   cdin......Inversion clouds
    !   cstc......cstc=clow change necessary in order to properly mark inv
    !             cloud height
    !   ccon......convc  =  ncols convective cloud cover in 3 hr. avrage
    !   cson......Shallow convective clouds
    !   cp........Specific heat of air (j/kg/k)
    !   gasr......Constant of dry air      (j/kg/k)
    !   mxrdcc....use maximum random converage for radiative conv. clouds
    !             constant logical mxrdcc = .true.
    !   lcnvl.....the lowest layer index where non-convective clouds can
    !             occur (ben says this should be 2 or more)
    !             constant lcnvl = 2
    !   lthncl....Minimum depth in mb of non-zero low level cloud
    !             consta lthncl=80
    !==========================================================================
    INTEGER      ,             INTENT(IN   ) :: ncols
    INTEGER      ,             INTENT(in   ) :: kmax
    INTEGER      ,             INTENT(in   ) :: nls
    REAL(KIND=r8),             INTENT(in   ) :: prsl  (ncols,kMax)
    REAL(KIND=r8),             INTENT(in   ) :: prsi  (ncols,kMax+1)
    REAL(KIND=r8),             INTENT(in   ) :: shmt  (ncols,kmax)
    REAL(KIND=r8),             INTENT(in   ) :: gice  (ncols,kmax)
    REAL(KIND=r8),             INTENT(in   ) :: gliq  (ncols,kmax)
    REAL(KIND=r8),             INTENT(in   ) :: grh   (ncols,kmax)
    REAL(KIND=r8),             INTENT(in   ) :: omg   (ncols,kmax)
    REAL(KIND=r8),             INTENT(in   ) :: ad_omg (ncols,kmax,2)
    REAL(KIND=r8),             INTENT(in   ) :: ad_tmp (ncols,kmax,2)
    REAL(KIND=r8),             INTENT(in   ) :: ad_grh (ncols,kmax,2)
    REAL(KIND=r8),             INTENT(in   ) :: gtmp_in  (ncols,kmax)
    REAL(KIND=r8) :: prec (ncols)
    REAL(KIND=r8),             INTENT(inout) :: css   (ncols,kmax)
    REAL(KIND=r8),             INTENT(inout) :: ccu   (ncols,kmax)
    REAL(KIND=r8),             INTENT(inout) :: cdin  (ncols,kmax)
    REAL(KIND=r8),             INTENT(inout) :: cstc  (ncols,kmax)
    REAL(KIND=r8),             INTENT(inout) :: ccon  (ncols,kmax)
    REAL(KIND=r8),             INTENT(inout) :: cson  (ncols,kmax)
    REAL(KIND=r8) :: auxl   (ncols,kmax)
    REAL(KIND=r8),             INTENT(in   ) :: convcin (ncols)
    REAL(KIND=r8),             INTENT(in   ) :: convtin (ncols)
    REAL(KIND=r8),             INTENT(in   ) :: convbin (ncols)
    INTEGER      ,             INTENT(in   ) :: lcnvl
    INTEGER      ,             INTENT(in   ) :: lthncl
    !
    !     dthdpm-----min (d(theta)/d(p))
    !     invb ------the base of inversion layer
    !     dthdpc-----criterion of (d(theta)/d(p))
    !
    REAL(KIND=r8),    PARAMETER :: dthdpc = -0.04_r8 !-0.4e-1_r8! dthdpc= -0.4e-1_r8-----criterion of (d(theta)/d(p))
    REAL(KIND=r8)    :: pll   (ncols,kmax)
    REAL(KIND=r8)    :: pii   (ncols,kmax+1)
    REAL(KIND=r8)    :: cauxil1(ncols,kmax)
    REAL(KIND=r8)    :: cauxil2(ncols,kmax)
    REAL(KIND=r8)    :: gtmp  (ncols,kmax)
    REAL(KIND=r8)    :: prcpt (ncols)
    REAL(KIND=r8)    :: convc (ncols)
    REAL(KIND=r8)    :: convt (ncols)
    REAL(KIND=r8)    :: convb (ncols)

    REAL(KIND=r8)    :: conv  (ncols)
    REAL(KIND=r8)    :: clow  (ncols)
    REAL(KIND=r8)    :: cmid  (ncols)
    REAL(KIND=r8)    :: chigh (ncols)! high clouds due to strong convection
    INTEGER          :: ktop  (ncols)
    INTEGER          :: kbot  (ncols)
    INTEGER          :: klow  (ncols)
    INTEGER          :: kmid  (ncols)
    INTEGER          :: khigh (ncols)! index of the high clouds due to strong convection
    REAL(KIND=r8)    :: pt    (ncols)
    REAL(KIND=r8)    :: cx    (ncols)
    REAL(KIND=r8)    :: cinv  (ncols)!low stratus associated with inversions
    REAL(KIND=r8)    :: omeganeg(ncols,kmax)
    REAL(KIND=r8)    :: delomg(ncols)
    REAL(KIND=r8)    :: dthdpm(ncols)
    INTEGER          :: invb  (ncols)
    REAL(KIND=r8)    :: SummOmeg(ncols)
    REAL(KIND=r8)    :: CounOmeg(ncols)
    REAL(KIND=r8)    :: AverOmeg(ncols)
    REAL(KIND=r8)    :: VariOmeg(ncols)
    REAL(KIND=r8)    :: normal(ncols,kmax)
    REAL(KIND=r8)    :: CLDF_PDF (ncols,kmax)
    REAL(KIND=r8)    :: CLDF_PDF_LY (ncols,kmax)
    REAL(KIND=r8)    :: frcliq(ncols,kmax)
    REAL(KIND=r8)    :: frcice(ncols,kmax)
    REAL(KIND=r8)    :: Count_convc(ncols)
    REAL(KIND=r8), PARAMETER :: f700p= 8.0e2_r8!7.5e2_r8! 7.0e2_r8
    REAL(KIND=r8), PARAMETER :: f400p= 4.0e2_r8
    REAL(KIND=r8), PARAMETER :: f6p67= 6.67e0_r8
    REAL(KIND=r8), PARAMETER :: f0p9 = 0.9e0_r8!Critical Relative Humidity   up level 400mb
    REAL(KIND=r8), PARAMETER :: f0p8 = 0.8e0_r8!Critical Relative Humidity down level 400mb 
    REAL(KIND=r8), PARAMETER :: f0p0 = 0.0e0_r8!Critical Relative Humidity down level 400mb 
    REAL(KIND=r8), PARAMETER :: f0p4 = 0.4e0_r8!Critical cloud due precipitation
    REAL(KIND=r8), PARAMETER :: f0p3 = 0.3e0_r8
    REAL(KIND=r8), PARAMETER :: f0p2 = 0.2e0_r8
    REAL(KIND=r8), PARAMETER :: f0p1 = 0.1e0_r8
    REAL(KIND=r8), PARAMETER :: f0p6 = 0.6e0_r8!Critical Relative Humidity to  stratus cloud
    REAL(KIND=r8), PARAMETER :: f1e4 = 1.0e+4_r8

    REAL(KIND=r8), PARAMETER :: f5m5 = 5.0e-5_r8!Critical vertical Velocity cb/s
    REAL(KIND=r8), PARAMETER :: f0m0 = 0.0e-0_r8!Critical vertical Velocity cb/s
    REAL(KIND=r8), PARAMETER :: f5m4 = 5.0e-4_r8!Critical vertical Velocity cb/s
    REAL(KIND=r8), PARAMETER :: f1m5 = 5.0e-5_r8!Critical vertical Velocity cb/s

    INTEGER :: i
    INTEGER :: k
    INTEGER :: kl
    INTEGER :: lon
    REAL(KIND=r8)    :: arcp
    REAL(KIND=r8)    :: dthdp
    REAL(KIND=r8)    :: thklow
    REAL(KIND=r8)    :: maxomeg

    REAL(KIND=r8)   , PARAMETER :: fp2457 = 0.2457_r8
    REAL(KIND=r8)   , PARAMETER :: fp1253 = 0.1253_r8
    REAL(KIND=r8)   , PARAMETER :: f8p0e3 = 8.0e3_r8  ! 1000 * 8

    DO i = 1, ncols
       pii   (i,kmax+1)= 0.0_r8
       conv  (i)= 0.0_r8
       clow  (i)= 0.0_r8
       cmid  (i)= 0.0_r8
       chigh (i)= 0.0_r8
       ktop  (i)= 0.0_r8
       kbot  (i)= 0.0_r8
       klow  (i)= 0.0_r8
       kmid  (i)= 0.0_r8
       khigh (i)= 0.0_r8
       pt    (i)= 0.0_r8
       cx    (i)= 0.0_r8
       cinv  (i)= 0.0_r8
       delomg(i)= 0.0_r8
       dthdpm(i)= 0.0_r8
       invb  (i)= 0.0_r8
       SummOmeg(i)= 0.0_r8
       CounOmeg(i)= 0.0_r8
       AverOmeg(i)= 0.0_r8
       VariOmeg(i)= 0.0_r8
       Count_convc(i)= 0.0_r8
       !convc(i)= 0.0_r8
      !READ(1,rec=irec)aux
      ! convt(i)=kMax-nls   ! index top cloud convective  
       convt(i)=convtin(i)   ! index top cloud convective  
       
      !READ(1,rec=irec)aux
      ! convb(i)=2          ! index botton cloud convective 
       convb(i)=convbin(i)          ! index botton cloud convective 

    END DO

    DO k = 1, kmax
       DO i = 1, ncols
          auxl(i,k)= 0.0_r8
          pii (i,k)= 0.0_r8
          pll (i,k) = 0.0_r8
          cauxil1(i,k) = 0.0_r8
          cauxil2(i,k) = 0.0_r8
          css(i,k)  = 0.0_r8
          ccu(i,k)  = 0.0_r8
          cdin(i,k) = 0.0_r8
          cstc(i,k) = 0.0_r8
          ccon(i,k) = 0.0_r8
          cson(i,k) = 0.0_r8
          normal(i,k) = 0.0_r8
          CLDF_PDF(i,k) = 0.0_r8
       END DO
    END DO

    CALL CloudFractionLayrson(nCols,kMax,grh,gliq ,gice,frcliq,frcice)

      DO k=1,kMax
         DO i=1,nCols
             CLDF_PDF_LY(i,k)=MAX(frcliq(i,k),frcice(i,k))
             !PRINT*,CLDF_PDF_LY(i,k),frcliq(i,k),frcice(i,k),gliq(i,k) ,gice(i,k)
         END DO
      END DO

      convc=0.0_r8
      Count_convc=0.0_r8
      DO k=1,kMax
         DO i=1,nCols
             If(CLDF_PDF_LY(i,k) > 0.05_r8  .and. CLDF_PDF_LY(i,k) <= 1.0_r8 )THEN
                convc(i)       = convc(i)       + MIN(CLDF_PDF_LY(i,k),1.0_r8)
                Count_convc(i) = Count_convc(i) + 1.0_r8
             END IF
         END DO
      END DO

      DO i=1,nCols
         !IF(convcin(i) > 0.0_r8 .and. SUM(CLDF_PDF_LY(i,1:kMax))/kMax < 0.01_r8 )THEN
         !    convc(i) = convcin(i)
         !    convc(i) = MAX(convc(i), 0.0e0_r8)
         !    convc(i) = MIN(convc(i), f0p8)
         !ELSE
         IF(Count_convc(i) /= 0.0_r8)THEN
             convc(i) = convc(i)/Count_convc(i)
             convc(i) = MAX(convc(i), 0.0e0_r8)
             convc(i) = MIN(convc(i), f0p8)
         ELSE
             convc(i) =0.0_r8
         END IF
         !END IF
      END DO
      DO k=1,kMax
         DO i=1,nCols
             CLDF_PDF_LY(i,k)=MIN(CLDF_PDF_LY(i,k),1.0_r8)
         END DO
      END DO

    CALL CLOUDS_GNO(ncols,kMax,prsl,gtmp_in,shmt,gice,gliq,CLDF_PDF)

    !--------------------------------------------------------
    !   CLOUD COVER, Cloud TOP-BOT FOR RADIATION (sub cldgen)
    !             DUE CONV PRECIPITATION
    !--------------------------------------------------------
    !   TO cldgen are necessary:
    !   a) cloud top and base   (convt, convb in cldgen)
    !   b) cloud amount is calculated convc (only rrr>0). It is calculate below.
    !     a+b used to defined high clouds due to strong convection
    !   prcpt=precipitation at each time step.(rrr) meters
    !   convt=ktop
    !   conbt=kbot (>=2) for radiation
    !*****************************************************************
    !   DO i = 1, ncols
    !      prcpt(i) = prcpt(i) - prcp1(i) + prcp3(i)!m
    !   END DO
    !DO i = 1, ncols 
    !   IF(prec(i) <0.02_r8)then
    !      prcpt(i) =0.0e0_r8
    !   ELSE
    !      prcpt(i) =prec(i)
    !   END IF
    !END DO
    ! 
    !IF(TRIM(iccon).EQ.'ARA')THEN
    !  DO i = 1, ncols 
    !     IF (prcpt(i) .GT. 0.0e0_r8) THEN
    !        convc(i) = fp2457 + fp1253 * LOG(prcpt(i) * f8p0e3)
    !        convc(i) = MAX(convc(i), 0.0e0_r8)
    !        convc(i) = MIN(convc(i), f0p8)
    !     END IF
    !  END DO
    !ELSE
    !  DO i = 1, ncols 
    !     IF (prcpt(i) .GT. 0.0e0_r8) THEN
    !        convc(i) = fp2457 + fp1253 * LOG(prcpt(i) * f8p0e3)
    !        convc(i) = MAX(convc(i), 0.0e0_r8)
    !        convc(i) = MIN(convc(i), f0p8)
    !     END IF
    !  END DO
    !END IF  

    DO kl = 1, kmax
       DO lon = 1, ncols 
          IF(omg(lon,kl) < 0.0e0_r8 ) THEN
             omeganeg(lon,kl)=omg(lon,kl)
          ELSE
             omeganeg(lon,kl)=0.0e0_r8
          END IF  
       END DO        
    END DO
    !
    !     the clouds generation scheme is based on j. slingo
    !     (1984 ecmwf workshop).  the scheme generates 4 type of clouds
    !     of convective, high, middle and low clouds.
    !
    DO kl = 1, kmax
       DO lon = 1,ncols
          pll(lon,kl) = prsl(lon,kl)/100.0_r8!gps(lon) * sigmid(kl)
       END DO
    END DO

    DO kl = 1, kmax+1
       DO lon = 1,ncols
          pii(lon,kl) = prsi(lon,kl)/100.0_r8!gps(lon) * sigmid(kl)
       END DO
    END DO
    DO kl = 1, kmax
       DO lon = 1,ncols
          SummOmeg(lon) =SummOmeg(lon)+omeganeg(lon,kl)
          CounOmeg(lon) = CounOmeg(lon)+1.0_r8
       END DO
    END DO
    DO lon = 1,ncols
       AverOmeg(lon) = SummOmeg(lon)/CounOmeg(lon)
    END DO
    DO kl = 1, kmax
       DO lon = 1,ncols
          VariOmeg(lon) = VariOmeg(lon) + ((omeganeg(lon,kl)-AverOmeg(lon))**2)
       END DO
    END DO
    DO lon = 1,ncols
       VariOmeg(lon) =SQRT(VariOmeg(lon)/CounOmeg(lon))
       IF(VariOmeg(lon) <=0.0)VariOmeg(lon)=0.0000001
    END DO
    DO lon = 1,ncols
        AverOmeg(lon)=MINVAL(omeganeg(lon,1:kmax))
    END DO
    !VariOmeg=VariOmeg!*0.25
    !lambda=2   
    !blognep=2.7182!e => base of the Napierian logarithm
    DO kl = 1, kmax
       DO lon = 1,ncols
          normal(lon,kl) = (1.0_r8/(VariOmeg(lon)*sqrt(2.0_r8*3.14_r8)))* &
          exp(-(1.0_r8/2.0_r8)*((omeganeg(lon,kl)-AverOmeg(lon))/(VariOmeg(lon)))**2)
 !         poisson(lon,kl) = (1.0_r8/(pll(lon,kl)*VariOmeg(lon)*sqrt(2.0_r8*3.14_r8)))* &
 !         exp(-((log(pll(lon,kl))-AverOmeg(lon))**2/(2.0_r8*VariOmeg(lon)*VariOmeg(lon))))
       END DO
    END DO

    maxomeg=MAXVAL(normal)
    DO kl = 1, kmax
       DO lon = 1,ncols
          normal(lon,kl) = normal(lon,kl)/maxomeg
       END DO
    END DO
    !******************************************************************************
    !
    !     initialization
    !
    !******************************************************************************
 !    maxomeg=MAXVAL(poisson)
    DO kl = 1, kmax
       DO lon = 1,ncols
          gtmp(lon,kl)=gtmp_in(lon,kl) - ad_tmp (lon,kl,2)
       END DO
    END DO

    DO  lon = 1,ncols
       !
       !clouds due to strong convection
       !
       conv(lon)  = convc(lon)

       clow(lon)  = 0.0_r8
       cinv(lon)  = 0.0_r8
       cmid(lon)  = 0.0_r8
       chigh(lon) = 0.0_r8
       !
       ! Check top cloud
       !       
       ktop(lon)  = INT( convt(lon)+0.5_r8 )
       IF (ktop(lon).LT.1.OR.ktop(lon).GT.kmax) ktop(lon)=1
       !
       ! Check botton cloud
       !       
       kbot(lon)  = INT(convb(lon)+0.5_r8)
       IF (kbot(lon).LT.1.OR.kbot(lon).GT.kmax) kbot(lon)=1
       !
       klow(lon)  = 1
       kmid(lon)  = 1
       khigh(lon) = 1  !index of the high clouds due to strong convection
       !
       !     1. define convective cloud
       !     done in subr-gwater
       !     cloud top and base are defined by kuo scheme: convt, convb
       !     cloud amount is calculated from precipitation rate : convc
       !     *** single layer clouds conputations start here, from bottom up
       !
       !     define high clouds due to strong convection
       !
       pt(lon) =pll(lon,ktop(lon))
       !
       ! Check the top cloud 
       ! if the top of the cloud is above 400mb and 
       ! the cloud due precipitation  is above 0.4 [Critical cloud due precipitation]
       !
       IF ((  pt(lon).LE.f400p).AND.(conv(lon).GE.f0p4)) THEN
          !
          ! high clouds due to strong convection
          !
          ! if the cloud cover exceeds a critical value of 0.3. 
          ! Starts the formation of high clouds.
          !
          !      f0p3 = 0.3e0_r8
          !
          chigh(lon) = 2.0_r8 * (conv(lon) - f0p3)
          chigh(lon) = MIN(chigh(lon),1.0_r8)
          !
          ! index of the high clouds due to strong convection
          !
          khigh(lon) = MIN(ktop(lon) + 1,kmax)
          !
          ! top cloud convection due  strong convection
          !
          ccon(lon,khigh(lon))=MIN(chigh(lon),1.0_r8)
       END IF
    END DO
    !******************************************************************************
    !
    !     compute low stratus associated with inversions, 
    !     based on ecwmf's scheme, 
    !     with lower criterion of d(theta)/d(p)
    !
    !******************************************************************************
    arcp = gasr / cp
    DO lon=1,ncols
       invb  (lon) =  MIN(kmax, kmax-nls)
       dthdpm(lon) = 0.0_r8
    END DO

    DO kl=2,kmax
       DO lon=1,ncols
          !
          ! compute low stratus associated with inversions
          ! Check Levels below 700 mb
          !
          IF (pll(lon,kl) .GT. f700p) THEN
             !
             ! Differency of potential temperature between 2 level [(k-1) - (k)]
             !
             dthdp = (gtmp(lon,kl-1)*(1000.0_r8/pll(lon,kl-1))**arcp &
                     -gtmp(lon,kl  )*(1000.0_r8/pll(lon,kl  ))**arcp)/&
                     (pll(lon,kl-1) - pll(lon,kl))
             !
             !  Conditional inversion
             ! lower criterion of d(theta)/d(p) < 0
             !
             ! theta(k-1)  -  theta(k)      -
             !------------------------- = -----   < 0   ===> stable
             !   P(k-1)    - P(k)           +
             !
             IF(dthdp.LT.0.0_r8) THEN
                !
                !change  index of the inversion layer
                !
                invb(lon) = MIN(kl-1,invb(lon))
                !
                ! dthdpc= -0.4e-1_r8-----criterion of (d(theta)/d(p))
                !
                ! theta(k-1)  -  theta(k)      -
                !------------------------- = -----   < -0.04   ===> near neutral
                !   P(k-1)    - P(k)           +
                !
                !
                IF(dthdp.LT.dthdpc) THEN
                   ! 
                   ! Checks for stability between 2 layers
                   !
                   !    k+1         <      k    
                   !
                   !d(thetap)/d(p)  < d(thetam)/d(p) 
                   !
                   IF(dthdp.LT.dthdpm(lon)) THEN
                      !
                      ! if the stability in the top layer is less than the 
                      ! stability in the bottom layer, the inversion
                      !
                      dthdpm(lon)=dthdp
                      !
                      !  index of the inversion layer
                      !
                      klow(lon)=kl
                   ENDIF
                ENDIF
             ENDIF
          ENDIF
       END DO
    END DO
    !
    !     klow change above necessary to mark inversion cloud height
    !     low stratus associated with inversions
    !
    DO lon=1,ncols
       !
       ! theta(k-1)  -  theta(k)      -
       !------------------------- = -----   < -0.04   ===> near neutral
       !   P(k-1)    - P(k)           +
       !
       !
       IF(dthdpm(lon).LT.dthdpc .AND. grh(lon,invb(lon))-ad_grh(lon,invb(lon),2).GT.f0p0)THEN
          !
          ! compute low stratus associated with inversions
          !          below 700 mb
          !
          ! dthdpc= -0.4e-1_r8-----criterion of (d(theta)/d(p))
          !
          !  f6p67 = 6.67e0_r8  Conversion factor temph to cloud fraction
          !
          ! low stratus associated with inversions
          !
          !cinv(lon) = - f6p67 * (dthdpm(lon) - dthdpc)
          cinv(lon) = MAX(CLDF_PDF_LY(lon,invb(lon)),CLDF_PDF(lon,invb(lon)))
          !
          cinv(lon) = MAX(cinv(lon),0.0_r8)
          cinv(lon) = MIN(cinv(lon),1.0_r8)
          !
          ! update low stratus associated with inversions whit relative humidity
          ! if  relative humidity < 0.8
          !
          IF (grh(lon,invb(lon))-ad_grh(lon,invb(lon),2) .LT. f0p0)THEN
             !
             ! low stratus associated with inversions
             !
             cinv(lon) = cinv(lon)*(1.0_r8-((f0p8 - grh(lon,invb(lon)))/f0p2))
          END IF

       ENDIF

       clow(lon)=cinv(lon)
       !
       !low stratus associated with inversions
       !
       cdin(lon,klow(lon))=MIN(cinv(lon),1.0_r8)

    END DO
    !
    !     clow change necessary in order to properly mark inv cloud height
    !
    !     main loop for cloud amount determination
    !
    DO kl = lcnvl+1, MIN(kmax, kmax-nls)
       DO lon = 1,ncols
          !---------------------------------------------------------------------------
          !
          !     general define cloud due to saturation
          !
          !---------------------------------------------------------------------------
          IF (pll(lon,kl) .GT. f400p) THEN
             !
             ! below 400 hPa (f400p= 4.0e2_r8)
             ! f0p8 = 0.8e0_r8  !Critical Relative Humidity down level 400mb 
             ! f0p2 = 0.2e0_r8  !Reference Critical Relative Humidity down level 400mb 
             !
             !
             !              Rh -Rhc
             !   cldsatn = ----------
             !                Rhr
             !
             cx(lon) = (grh(lon,kl) - f0p8) / f0p2
          ELSE
             !
             ! below 400 hPa (f400p= 4.0e2_r8)
             ! f0p9 = 0.9e0_r8 !Critical Relative Humidity   up level 400mb
             ! f0p1 = 0.1e0_r8 !Reference Critical Relative Humidity down level 400mb 
             !
             !              Rh -Rhc
             !   cldsatn = ----------
             !                Rhr
             !
             cx(lon) = (grh(lon,kl) - f0p9) / f0p1
          END IF
          cx(lon) = MAX(CLDF_PDF_LY(lon,kl),CLDF_PDF(lon,kl))
          !
          ! cldsat = cldsatn * cldsatn
          !
          cx(lon) = (MAX(cx(lon), 0.0_r8))! ** 2
          !
          ! checking the saturation cloud (cldsatn) thresholds 0-1
          !
          cx(lon)      =  MIN(cx(lon), 1.0_r8)
          auxl(lon,kl) =  cx(lon)
          !---------------------------------------------------------------------------
          !
          !     start vertical process from bottom to top
          !
          !---------------------------------------------------------------------------
          IF (pll(lon,kl) .GT. f700p) THEN
             !
             !     below 700 hPa 
             !
             !     2. define low cloud ***
             !     low cloud is defined one layer thick ranging from layer 3 to 700mb
             !     there are two type possible generating mechanisms. due boundary
             !     t inversion type and associated with vertical motion.
             !
             !     define low super satuated clouds but adjusted by vertical motion
             !
             !IF (omg(lon,kl) .GE. f5m5) THEN
             IF ((omg(lon,kl) - ad_omg(lon,kl,2)) .GE. f0m0 ) THEN
                !  
                !  vertical speed (downward mass flow) 
                !
                !  f5m5 = 5.0e-5_r8   !Max and Min Critical vertical Velocity cb/s
                !
                !                      no satuated cloud  
                !
                cx(lon)      = 0.0_r8
                auxl(lon,kl) = cx(lon)
                !
             !ELSE IF (omg(lon,kl) .GE. -f5m5) THEN
             ELSE IF ((omg(lon,kl) - ad_omg(lon,kl,2)) .GE. -f5m5 .AND. &
                      (omg(lon,kl) - ad_omg(lon,kl,2)) .LT.  f0m0 ) THEN

                !
                ! vertical speed (upward-downward mass flow)
                !      (   -f5m5  <-->  f5m5  )
                !
                !  f5m5 = 5.0e-5_r8   !Max and Min Critical vertical Velocity cb/s
                !
                ! satuated cloud  
                !
                ! f1e4 = 1.0e+4_r8   ---> 10000
                !ad_omg
                !delomg(lon) = (omg(lon,kl) + f5m5) * f1e4
                delomg(lon) = MIN((omg(lon,kl) - ad_omg(lon,kl,2)),0.0_r8) * f1e4
                !PRINT*,delomg(lon)
                !
                !
                cx(lon) = cx(lon) * (1.0_r8 - delomg(lon) * delomg(lon))
                !cx(lon) = MAX(CLDF_PDF_LY(lon,kl),CLDF_PDF(lon,kl))
                !
                !
                auxl(lon,kl)=cx(lon)
                !
             END IF
             !
             ! update  low cloud due to saturation
             !
             IF (cx(lon) .GT. clow(lon)) THEN
                klow(lon) = kl
                clow(lon) = cx(lon)
             END IF

          ELSE IF (pll(lon,kl) .LE. f700p .and. pll(lon,kl) .GT. f400p) THEN
             !
             !     3. define middle cloud ***
             !     middle cloud is defined one layer thick between 700 and 400 mb.
             !
             !     define middle clouds only in supersaturate type
             !
             IF (cx(lon) .GT. cmid(lon)) THEN
                kmid(lon) = kl
                cmid(lon) = cx(lon)
             END IF
          ELSE
             !
             !     4. define high cloud
             !     high cloud is defined only one layer thick from 400 mb and up.
             !
             !     define high clouds due to satuation
             !
             IF (cx(lon) .GT. chigh(lon)) THEN
                khigh(lon) = kl
                chigh(lon) = cx(lon)
             END IF
             !
             !     end of vertical computation
             !
          END IF
       END DO
    END DO



    DO lon = 1,ncols
       ! 
       ! css --> supersatuation cloud cover fraction
       !
       css (lon,kmax-khigh(lon)+1) = MIN(chigh(lon),1.0_r8)
       !
       ! cstc --> change necessary in order to properly mark inversion cloud height
       !
       cstc(lon,khigh(lon))        = MIN(chigh(lon),1.0_r8)
       ! 
       ! css --> supersatuation cloud cover fraction
       !
       css (lon,kmax-kmid(lon)+1)  = MIN(cmid(lon),1.0_r8)
       !
       ! cstc --> change necessary in order to properly mark inversion cloud height
       !
       cstc(lon,kmid(lon))         = MIN(cmid(lon),1.0_r8)
       !
       !     for very thin low cloud adding its thickness
       !     pressure thickness of low cloud layer
       !
       IF(klow(lon).GE.lcnvl) THEN
          !
          !pressure thickness of low cloud layer
          !
          thklow=0.5_r8*(pll(lon,klow(lon)-1)-pll(lon,klow(lon)+1))
          !
          !lthncl=80 => Minimum depth in mb of non-zero low level cloud  constant
          !
          IF(thklow.LE.REAL(lthncl,r8)) THEN
             ! 
             ! css --> supersatuation cloud cover fraction
             !
             css (lon,kmax-klow(lon)) = MIN(clow(lon),1.0_r8)

             cson(lon,klow(lon)+1)     = MIN(clow(lon),1.0_r8)
             !
             ! cstc --> change necessary in order to properly mark inversion cloud height
             !
             cstc(lon,klow(lon)+1)     =MIN(clow(lon),1.0_r8)
          ENDIF
          ! 
          ! css --> supersatuation cloud cover fraction
          !
          css(lon,kmax-klow(lon)+1) = MIN(clow(lon),1.0_r8)
 
          cson(lon,klow(lon))       = MIN(clow(lon),1.0_r8)
          !
          ! cstc --> change necessary in order to properly mark inversion cloud height
          !
          cstc(lon,klow(lon))       = MIN(clow(lon),1.0_r8)
       ENDIF
    END DO

    !NEW CODE-------------------------------------------------------------
    DO kl =1,kMax
       DO lon = 1,ncols
          IF( kl >= kbot(lon) .AND. kl <=ktop(lon)) THEN
             !       DO kl = kbot(lon), ktop(lon)
             IF(omg   (lon,kl) < -2.0E-004_r8)THEN
                ccu(lon,kmax-kl+1) = MAX(CLDF_PDF_LY(lon,kl),CLDF_PDF(lon,kl),conv(lon)*normal(lon,kl))
                !
                ! cloud convection
                !
                ccon(lon,kl)       = MAX(CLDF_PDF_LY(lon,kl),CLDF_PDF(lon,kl),conv(lon)*normal(lon,kl))
             ELSE
                ccu(lon,kmax-kl+1) = MAX(CLDF_PDF_LY(lon,kl),conv(lon)*normal(lon,kl))
                !
                ! cloud convection
                !
                ccon(lon,kl)       = MAX(CLDF_PDF_LY(lon,kl),conv(lon)*normal(lon,kl))
             END IF
             css(lon,kmax-kl+1) = MIN(1.0_r8,css(lon,kmax-kl+1))
          END IF
       END DO
    END DO

    DO kl =1,kMax
       DO lon = 1,ncols
          IF( kl >= kbot(lon) .AND. kl <=ktop(lon)) THEN
             cauxil1(lon,kl)=css(lon,kl)
             cauxil2(lon,kl)=ccu(lon,kl)
             css(lon,kl)= cauxil1(lon,kl)
             ccu(lon,kl)= cauxil2(lon,kl) 
          END IF
       END DO
    END DO

    RETURN
  END SUBROUTINE cldgn4


  !
  ! $Header$
  !
  !
  !================================================================================
  !
!  SUBROUTINE CLOUDS_GNO(nCols,kMax,R,RS,QSUB,CLDF)
  SUBROUTINE CLOUDS_GNO(nCols,kMax,prsl,t3,q3,gicep,gliqp,CLDF)

    IMPLICIT NONE
    !     
    !--------------------------------------------------------------------------------
    !
    ! Inputs:
    !
    !  kMax----------: Number of vertical levels
    !  R--------kMax-: Domain-averaged mixing ratio of total water 
    !  RS-------kMax-: Mean saturation humidity mixing ratio within the gridbox
    !  QSUB-----kMax-: Mixing ratio of condensed water within clouds associated
    !                with SUBGRID-SCALE condensation processes (here, it is
    !                predicted by the convection scheme)
    ! Outputs:
    !
    !  PTCONV-----kMax-: Point convectif = TRUE
    !  RATQSC-----kMax-: Normalized width of the distribution
    !  CLDF-----kMax-: Fraction nuageuse
    !
    !--------------------------------------------------------------------------------


    INTEGER      , INTENT(IN   ) :: nCols
    INTEGER      , INTENT(IN   ) :: kMax
    REAL(KIND=r8), INTENT(IN   ) :: prsl(nCols,kMax)
    REAL(KIND=r8), INTENT(IN   ) :: t3(nCols,kMax)
    REAL(KIND=r8), INTENT(IN   ) :: q3(nCols,kMax)
    REAL(KIND=r8), INTENT(IN   ) :: gicep(nCols,kMax)
    REAL(KIND=r8), INTENT(IN   ) :: gliqp(nCols,kMax)
    REAL(KIND=r8), INTENT(OUT  ) :: CLDF(nCols,kMax)
    ! -- parameters controlling the iteration:
    ! --    nmax    : maximum nb of iterations (hopefully never reached)
    ! --    epsilon : accuracy of the numerical resolution 
    ! --    Omega    : v-value above which we use an asymptotic expression for ERF(v)

    INTEGER     , PARAMETER :: nmax = 10
    REAL(KIND=r8),PARAMETER :: epsilon= 0.02_r8
    REAL(KIND=r8),PARAMETER :: Omega0= 2.0_r8 ! where omega is a specific cutoff value for gama/ksqrt(2)

    REAL(KIND=r8),PARAMETER :: min_mu  =  1.e-12_r8
    REAL(KIND=r8),PARAMETER :: min_Q = 1.e-12_r8 
    REAL(KIND=r8) :: Omega(nCols)
    REAL(KIND=r8) :: Coef_Miu(nCols)
    REAL(KIND=r8) :: qsat
    REAL(KIND=r8) :: delta(nCols)
    REAL(KIND=r8) :: beta(nCols) 
    REAL(KIND=r8) :: zu2
    REAL(KIND=r8) :: zv2
    REAL(KIND=r8) :: Coef_k(nCols)
    REAL(KIND=r8) :: aux(nCols)
    REAL(KIND=r8) :: coeff
    REAL(KIND=r8) :: BLOCK1
    REAL(KIND=r8) :: dist_gk
    REAL(KIND=r8) :: fprime
    REAL(KIND=r8) :: det
    REAL(KIND=r8) :: pi
    REAL(KIND=r8) :: u
    REAL(KIND=r8) :: v
    REAL(KIND=r8) :: erfcu
    REAL(KIND=r8) :: erfcv
    REAL(KIND=r8) :: Coef_k1
    REAL(KIND=r8) :: Coef_k2
    REAL(KIND=r8) :: erf
    REAL(KIND=r8) :: hsqrtlog_2
    REAL(KIND=r8) :: v2
    REAL(KIND=r8) :: sqrtpi
    REAL(KIND=r8) :: sqrt2
    REAL(KIND=r8) :: zx1
    REAL(KIND=r8) :: zx2
    REAL(KIND=r8) :: exdel
    ! lconv = true si le calcul a converge (entre autre si qsub < min_q)
    LOGICAL       :: lconv(nCols)
    LOGICAL       :: PTCONV(nCols,kMax)
    REAL(KIND=r8) :: RATQSC(nCols,kMax) !Normalized width of the distribution
    REAL(KIND=r8) :: R     (nCols,kMax)
    REAL(KIND=r8) :: RS    (nCols,kMax)
    REAL(KIND=r8) :: QSUB  (nCols,kMax)
    REAL(KIND=r8) :: qes(nCols,kMax) 
    REAL(KIND=r8) :: tsp(nCols,kMax) 
    INTEGER       :: i
    INTEGER       :: K
    INTEGER       :: n
    INTEGER       :: m


    CALL findsp (nCols,kMax, q3, t3, prsl, tsp, qes)

    DO  K = 1, kMax !500
        DO i=1,nCols ! vector
           R     (i,k)=  q3     (i,k)  !  R--------kMax-: Domain-averaged mixing ratio of total water 
           RS    (i,k)=  qes    (i,k)  !  RS-------kMax-: Mean saturation humidity mixing ratio within the gridbox
           QSUB  (i,k)=  gliqp  (i,k)+gicep(i,k)   !  QSUB-----kMax-: Mixing ratio of condensed water within clouds associated
                                       !  with SUBGRID-SCALE condensation processes (here, it is
                                       !  predicted by the convection scheme)
        END DO
    END DO
    !cdir arraycomb
    cldf  (1:nCols,1:kMax)=0.0_r8        ! cym
    ratqsc(1:nCols,1:kMax)=0.0_r8        ! Normalized width of the distribution
    ptconv(1:nCols,1:kMax)=.FALSE.       ! Point convect if = TRUE
    !cdir end arraycomb

    pi = ACOS(-1.0_r8)

    sqrtpi=SQRT(pi)

    sqrt2=SQRT(2.0_r8)
    !
    !                         _______
    !              1   /\    /       \
    !hsqrtlog_2 = ---    \  / log (2)
    !              2      \/
    !
    hsqrtlog_2=0.5_r8*SQRT(LOG(2.0_r8))
    DO  K = 1, kMax !500

       DO i=1,nCols ! vector
          Coef_Miu(i)    = R(i,K)           ! Domain-averaged mixing ratio of total water assumed to be a random variable characterized by a mean value
          Coef_Miu(i)    = MAX(Coef_Miu(i),min_mu)! Domain-averaged mixing ratio of total water assumed to be a random variable characterized by a mean value
          qsat     = RS(i,K)          ! Mean saturation humidity mixing ratio within the gridbox
          qsat     = MAX(qsat,min_mu) ! Mean saturation humidity mixing ratio within the gridbox
          !
          ! Log relative humidity
          !             _          _               _          _  
          !            |   X - miu  |             |      miu   | 
          ! gama = log | -----------| ~ delta =   | -----------| 
          !            |_  X - qs  _|             |_     qs   _| 
          !
          ! Hipothesis the the second and third statistical moments can be diagnosed 
          !  i) fronm the in-cloud water content predicted by the convection scheme as a resultof convective activity that place the domain
          ! ii) From the degree of saturation of the large scale  environmental, and 
          !iii) by insisting that the total water mixing ratio values ar posiive
          !     so X = 0
          !
          delta(i) = LOG(Coef_Miu(i)/qsat)  ! Napierian logarithm of the rate mixing ratio of total water  and  saturation mixing ratio within the gridbox (relative Humidty)
          !                                   enddo ! vector
          !
          ! ***          There is no subgrid-scale condensation;        ***
          ! ***   the scheme becomes equivalent to an "all-or-nothing"  *** 
          ! ***             large-scale condensation scheme.            ***
          !

          !
          ! ***     Some condensation is produced at the subgrid-scale       ***
          ! ***                                                              ***
          ! ***       PDF = generalized log-normal distribution (GNO)        ***
          ! ***   (k<0 because a lower bound is considered for the PDF)      ***
          ! ***                                                              ***
          ! ***  -> Determine x (the parameter k of the GNO PDF) such        ***
          ! ***  that the contribution of subgrid-scale processes to         ***
          ! ***  the in-cloud water content is equal to QSUB(K)              ***
          ! ***  (equations (13), (14), (15) + Appendix B of the paper)      ***
          ! ***  SANDRINE BONY  AND KERRYA. EMANUEL, November 2001,          ***
          ! ***  JOURNAL OF THE ATMOSPHERIC SCIENCES,                        ***
          ! ***  A Parameterization of the Cloudiness Associated with Cumulus***
          ! ***  Convection; EvaluationUsing TOGA COARE Data
          ! ***                                                              ***
          ! ***    Here, an iterative method is used for this purpose        ***
          ! ***    (other numerical methods might be more efficient)         ***
          ! ***                                                              ***
          ! ***          NB: the "error function" is called ERF              ***
          ! ***                 (ERF in double precision)                   ***
          !

          !  On commence par eliminer les cas pour lesquels on n'a pas
          !  suffisamment d'eau nuageuse.

          ! check the Mixing ratio of condensed water within clouds
          !  
          IF ( QSUB(i,K) .LT. min_Q ) THEN
             ! 
             !No Mixing ratio of condensed water within clouds
             !
             ptconv(i,k)=.FALSE. !Point convect if = TRUE
             ratqsc(i,k)=0.0_r8  !Normalized width of the distribution
             lconv(i)  = .TRUE.  ! calculation converges (among other things if qsub <min_q)
             !
             !   Rien on a deja initialise
             !
          ELSE 
             ! 
             !Yes have Mixing ratio of condensed water within clouds
             !
             lconv(i)  = .FALSE. !not  calculation converges (among other things if qsub <min_q)
             Omega (i)  =  Omega0  !   Omega0= 2.0_r8 -->v-value above which we use an asymptotic expression for ERF(v)
             !
             !
             ! delta= log(RH) === > Log relative humidity
             !
             !                     _          _  
             !        ql          |            | 
             !beta = ----  +  exp |  - log(RH) | 
             !        q           |_          _| 
             !
             !print*,delta(i),MIN(0.0_r8,delta(i)),Coef_Miu(i)/qsat, EXP( -MIN(0.0_r8,delta(i)) )
             !
             ! Eq. B1 (SANDRINE BONY  AND KERRYA. EMANUEL, November 2001)
             !
             beta(i) = QSUB(i,K)/Coef_Miu(i) + EXP( -MIN(0.0_r8,delta(i)) )
             !
             ! --  roots of equation v > Omega: v-value above which we use an asymptotic expression for ERF(v)
             ! delta = Log of relative humidity
             !
             det = delta(i) + Omega(i)*Omega(i)

             IF (det.LE.0.0_r8) Omega(i) = Omega0 + 1.0_r8

             det = delta(i) + Omega(i)*Omega(i)

             IF (det.LE.0.0_r8) THEN
                Coef_k(i) = -0.0001_r8
             ELSE 
                !
                !sqrt2=SQRT(2.0_r8)
                !
                !                          _______
                !                   /\    /       \
                !zx1=      -Omega *   \  /     2     
                !                      \/
                !
                zx1   = -sqrt2*Omega(i)
                !
                ! log(1) = 0
                !
                ! log(a + b) = log (a) + log (1 + b/a)
                !
                ! log(a - b) = log (a) + log (1 - b/a)
                !
                !                           ________________________
                !                  /\      /          log (RH)      \
                !zx2        =        \    / 1.0   + ----------------
                !                     \  /           Omega(i)*Omega(i)
                !                      \/
                !
                zx2   = SQRT(1.0_r8 + delta(i)/(Omega(i)*Omega(i)))
                !                          _______
                !                   /\    /       \
                !zx1=      -Omega *   \  /     2     
                !                      \/
                !
                !
                !                           ________________________
                !                  /\      /          log (RH)      \
                !zx2        =        \    / 1.0   + ----------------
                !                     \  /           Omega(i)*Omega(i)
                !                      \/
                !
                ! Coef_k1   = zx1*(1.0_r8 - zx2)
                !
                ! Coef_k1   = zx1 - zx1*zx2)
                !                          _______                      _______                  ________________________
                !                   /\    /       \              /\    /       \       /\      /             log (RH)    \
                !Coef_k1=  -Omega *   \  /     2        +Omega *   \  /     2     *      \    / 1.0   + ----------------
                !                      \/                           \/                    \  /          Omega(i)*Omega(i)
                !                                                                          \/
                !
                !                          _______                   ____________________________
                !                   /\    /       \         /\      /                            \
                !Coef_k1=  -Omega *   \  /     2        +     \    / 2.0 Omega(i)^2  + 2* log (RH)
                !                      \/                      \  /                      
                !                                               \/
                !

                Coef_k1   = zx1*(1.0_r8 - zx2)

                Coef_k2   = zx1*(1.0_r8 + zx2)

                Coef_k(i) = 1.01_r8 * Coef_k1

                IF ( Coef_k1 .GE. 0.0_r8 ) Coef_k(i) = 0.5_r8*Coef_k2

             ENDIF
             !                         _______
             !              1   /\    /       \
             !hsqrtlog_2 = ---    \  / log (2)
             !              2      \/
             !
             IF (delta(i).LT.0.0_r8) Coef_k(i) = -hsqrtlog_2

          ENDIF

       ENDDO       ! vector

       !----------------------------------------------------------------------
       !   Debut des nmax iterations pour trouver la solution.
       !----------------------------------------------------------------------

       DO n = 1, nmax 

          DO i=1,nCols ! vector
             IF (.NOT.lconv(i)) THEN

                ! if .lconv(i) ==false the calculation converges (among other things if qsub <min_q)

                u = delta(i)/(Coef_k(i)*sqrt2) + Coef_k(i)/(2.0_r8*sqrt2)

                v = delta(i)/(Coef_k(i)*sqrt2) - Coef_k(i)/(2.0_r8*sqrt2)

                v2 = v*v

                IF ( v .GT. Omega(i) ) THEN 

                   IF (     ABS(u)  .GT. Omega(i) &
                        .AND.  delta(i) .LT. 0.0_r8 ) THEN
                      !
                      ! -- use asymptotic expression of erf for u and v large:
                      ! ( -> analytic solution for Coef_k )
                      !
                      exdel=beta(i)*EXP(delta(i))
                      !
                      aux(i) = 2.0_r8*delta(i)*(1.0_r8-exdel)/(1.0_r8+exdel)
                      !
                      IF (aux(i).LT.0.0_r8) THEN
                         !                print*,'AUX(',i,',',k,')<0',aux(i),delta(i),beta(i)
                         aux(i)=0.0_r8
                      ENDIF
                      Coef_k(i) = -SQRT(aux(i))
                      !
                      !Eq. 13 of the SANDRINE BONY  AND KERRYA. EMANUEL, November 2001, 
                      !
                      BLOCK1 = EXP(-v*v) / v / sqrtpi
                      !
                      dist_gk   = 0.0_r8
                      fprime    = 1.0_r8
                      !
                   ELSE
                      !
                      ! -- erfv -> 1.0_r8, use an asymptotic expression of erfv for v large:
                      !
                      erfcu = 1.0_r8 - erf_cldf(u)
                      !
                      !  !!! ATTENTION : rajout d'un seuil pour l'exponentiel
                      aux(i) = sqrtpi*erfcu*EXP(MIN(v2,100.0_r8))
                      coeff  = 1.0_r8 - 0.5_r8/(v2) + 0.75_r8/(v2*v2)
                      !
                      !Eq. 12 of the SANDRINE BONY  AND KERRYA. EMANUEL, November 2001, 
                      !
                      BLOCK1 = coeff * EXP(-v2) / v / sqrtpi
                      !
                      dist_gk = v * aux(i) / coeff - beta(i)
                      !
                      fprime = 2.0_r8 / Coef_k(i) * (v2)   &
                           * ( EXP(-delta(i)) - u * aux(i) / coeff ) &
                           / coeff
                      !
                   ENDIF ! ABS(u)

                ELSE
                   !
                   ! -- general case:
                   !
                   erfcu = 1.0_r8-erf_cldf(u)
                   erfcv = 1.0_r8-erf_cldf(v)
                   !
                   !Eq. 13 of the SANDRINE BONY  AND KERRYA. EMANUEL, November 2001, 
                   !
                   BLOCK1 = erfcv

                   dist_gk = erfcu / erfcv - beta(i)
                   !
                   zu2=u*u
                   zv2=v2
                   IF(zu2.GT.20.0_r8.OR. zv2.GT.20.0_r8) THEN
                      !     print*,'ATTENTION !!! Coef_k(',i,') =', Coef_k(i)
                      !     print*,'ATTENTION !!! nCols,ND,R,RS,QSUB,PTCONV,RATQSC,CLDF',
                      !     .nCols,ND,R(i,k),RS(i,k),QSUB(i,k),PTCONV(i,k),RATQSC(i,k),
                      !     .CLDF(i,k)
                      !              print*,'ATTENTION !!! zu2 zv2 =',zu2(i),zv2(i)
                      zu2=20.0_r8
                      zv2=20.0_r8
                      fprime = 0.0_r8
                   ELSE
                      fprime = 2.0_r8 /sqrtpi/Coef_k(i) /(erfcv*erfcv) &
                             * (   erfcv*v*EXP(-zu2) - erfcu*u*EXP(-zv2) )
                   ENDIF
                ENDIF ! x

                ! -- test numerical convergence:

                !          if (beta(i).lt.1.e-10) then
                !              print*,'avant test ',i,k,lconv(i),u(i),v(i),beta(i)
                !              stop
                !          endif
                IF (ABS(fprime).LT.1.e-11_r8) THEN
                   !              print*,'avant test fprime<.e-11 '
                   !     s        ,i,k,lconv(i),u(i),v(i),beta(i),fprime(i)
                   !              print*,'nCols,ND,R,RS,QSUB',
                   !     s        nCols,ND,R(i,k),rs(i,k),qsub(i,k)
                   fprime=SIGN(1.e-11_r8,fprime)
                ENDIF


                IF ( ABS(dist_gk/beta(i)) .LT. epsilon ) THEN 
                   !           print*,'v-u **2',(v(i)-u(i))**2
                   !           print*,'exp v-u **2',exp((v(i)-u(i))**2)
                   ptconv(i,K) = .TRUE.  ! Point convect if = TRUE
                   lconv(i)=.TRUE.
                   !  borne pour l'exponentielle
                   !
                   ! ratqsc, Normalized width of the distribution
                   !
                   ratqsc(i,k)=MIN(2.0_r8*(v-u)*(v-u),20.0_r8)  
                   ratqsc(i,k)=SQRT(EXP(ratqsc(i,k))-1.0_r8)
                   CLDF(i,K) = MIN(MAX(0.5_r8 * BLOCK1,0.01_r8),1.0_r8)
                   IF( CLDF(i,K) == 0.01_r8)CLDF(i,K) =0.0_r8
                ELSE
                   Coef_k(i) = Coef_k(i) - dist_gk/fprime
                ENDIF
                !         print*,'apres test ',i,k,lconv(i)

             ENDIF ! lconv
          ENDDO       ! vector

          !----------------------------------------------------------------------
          !   Fin des nmax iterations pour trouver la solution.
       ENDDO ! n
       !----------------------------------------------------------------------

    END DO !500    CONTINUE  ! K

    RETURN
  END SUBROUTINE CLOUDS_GNO


SUBROUTINE CloudFractionLayrson(nCols,kMax,grh,gliq ,gice,frcliq,frcice)
 IMPLICIT  NONE
 !
 ! LOCAL VARIABEL
 !
 INTEGER      , INTENT(IN   ) :: nCols,kMax
 REAL(KIND=r8), INTENT(IN   ) :: grh    (nCols,kMax)
 REAL(KIND=r8), INTENT(IN   ) :: gliq   (nCols,kMax)
 REAL(KIND=r8), INTENT(IN   ) :: gice   (nCols,kMax)
 REAL(KIND=r8), INTENT(OUT  ) :: frcliq (nCols,kMax)
 REAL(KIND=r8), INTENT(OUT  ) :: frcice (nCols,kMax)
 INTEGER, PARAMETER :: nCloudClass=15
 INTEGER, PARAMETER :: nsoilay=1000
 INTEGER, PARAMETER :: ibMax=1
 INTEGER, PARAMETER :: jbMax=1
 ! 
 ! beta_lw_liq   o parametro de forma
 !
 REAL(KIND=r8), PARAMETER    :: beta_lw_liq(nCloudClass)     = RESHAPE ( (/ &
 !  beta_lw_liq        !                                                  beta_lw_liq
   1.000_r8, &    ! classe  1:                                            0.962
   1.800_r8, &    ! classe  2:                                            0.961
   2.000_r8, &    ! classe  3:                                            0.966
   2.700_r8, &    ! classe  4:                                            0.966
   2.100_r8, &    ! classe  5:                                            0.965
   1.500_r8, &    ! classe  6:                                            0.960
   1.400_r8, &    ! classe  7:                                            0.950
   1.800_r8, &    ! classe  8:                                            0.960
   1.300_r8, &    ! classe  9:                                            0.962
   1.000_r8, &    ! classe 10:                                            0.952
   1.200_r8, &    ! classe 11:                                            0.970
   1.100_r8, &    ! classe 12:                                            0.950
   1.000_r8, &    ! classe 13:                                            0.914
   2.000_r8, &    ! classe 14:                                            0.970
   1.900_r8  &    ! classe 15:                                            0.970
   /), (/15/) )!---->  cloud systems  
 ! 
 ! beta_up_ice   o parametro de forma
 !
 REAL(KIND=r8), PARAMETER    :: beta_up_ice(nCloudClass)     = RESHAPE ( (/ &
  !  beta_up_ice        !                                                 beta_up_ice
   1.200_r8, &    ! classe  1:                                            0.962
   2.300_r8, &    ! classe  2:                                            0.961
   2.300_r8, &    ! classe  3:                                            0.966
   2.400_r8, &    ! classe  4:                                            0.966
   2.200_r8, &    ! classe  5:                                            0.965
   1.300_r8, &    ! classe  6:                                            0.960
   1.300_r8, &    ! classe  7:                                            0.950
   2.500_r8, &    ! classe  8:                                            0.960
   1.500_r8, &    ! classe  9:                                            0.962
   1.200_r8, &    ! classe 10:                                            0.952
   1.200_r8, &    ! classe 11:                                            0.970
   1.100_r8, &    ! classe 12:                                            0.950
   1.000_r8, &    ! classe 13:                                            0.914
   2.100_r8, &    ! classe 14:                                            0.970
   2.000_r8  &    ! classe 15:                                            0.970
   /), (/15/) )!---->   cloud systems  
   !
   ! theta_lw_liq  o parametro de escala 
   !
 REAL(KIND=r8), PARAMETER    :: theta_lw_liquid_RH(nCloudClass)     = RESHAPE ( (/ &
  !  theta_lw_liq        !                                                theta_lw_liq
    0.90_r8, &    ! classe  1:                                            0.962
    0.961_r8, &    ! classe  2:                                            0.961
    0.966_r8, &    ! classe  3:                                            0.966
    0.966_r8, &    ! classe  4:                                            0.966
    0.965_r8, &    ! classe  5:                                            0.965
    0.960_r8, &    ! classe  6:                                            0.960
    0.950_r8, &    ! classe  7:                                            0.950
    0.960_r8, &    ! classe  8:                                            0.960
    0.962_r8, &    ! classe  9:                                            0.962
    0.952_r8, &    ! classe 10:                                            0.952
    0.970_r8, &    ! classe 11:                                            0.970
    0.950_r8, &    ! classe 12:                                            0.950
    0.914_r8, &    ! classe 13:                                            0.914
    0.970_r8, &    ! classe 14:                                            0.970
    0.970_r8  &    ! classe 15:                                            0.970
   /), (/15/) )!---->  cloud systems  
   !
   ! theta_up_ice  o parametro de escala 
   !
 REAL(KIND=r8), PARAMETER    :: theta_up_ice_RH(nCloudClass)     = RESHAPE ( (/ &
  !  theta_up_ice        !                                                theta_up_ice
   0.90_r8, &     ! classe  1:                                            0.962
   0.961_r8, &    ! classe  2:                                            0.961
   0.966_r8, &    ! classe  3:                                            0.966
   0.966_r8, &    ! classe  4:                                            0.966
   0.965_r8, &    ! classe  5:                                            0.965
   0.960_r8, &    ! classe  6:                                            0.960
   0.950_r8, &    ! classe  7:                                            0.950
   0.960_r8, &    ! classe  8:                                            0.960
   0.962_r8, &    ! classe  9:                                            0.962
   0.952_r8, &    ! classe 10:                                            0.952
   0.970_r8, &    ! classe 11:                                            0.970
   0.950_r8, &    ! classe 12:                                            0.950
   0.914_r8, &    ! classe 13:                                            0.914
   0.970_r8, &    ! classe 14:                                            0.970
   0.970_r8  &    ! classe 15:                                            0.970
   /), (/15/) )!---->  cloud systems  
   ! 
   ! delta_lw_liq  o parametro de localizao
   !
 REAL(KIND=r8), PARAMETER    :: delta_lw_liq(nCloudClass)     = RESHAPE ( (/ &
  !  delta_lw_liq        !                                                delta_lw_liq
   0.0e-0_r8, &   ! classe  1:                                            0.962
   1.000_r8, &    ! classe  2:                                            0.961
   1.000_r8, &    ! classe  3:                                            0.966
   1.000_r8, &    ! classe  4:                                            0.966
   1.000_r8, &    ! classe  5:                                            0.965
   1.000_r8, &    ! classe  6:                                            0.960
   1.000_r8, &    ! classe  7:                                            0.950
   1.000_r8, &    ! classe  8:                                            0.960
   1.000_r8, &    ! classe  9:                                            0.962
   1.000_r8, &    ! classe 10:                                            0.952
   1.000_r8, &    ! classe 11:                                            0.970
   1.000_r8, &    ! classe 12:                                            0.950
   1.000_r8, &    ! classe 13:                                            0.914
   1.000_r8, &    ! classe 14:                                            0.970
   1.000_r8  &    ! classe 15:                                            0.970
   /), (/15/) )!---->   cloud systems  
   ! 
   ! delta_up_ice  o parametro de localizao
   !
 REAL(KIND=r8), PARAMETER    :: delta_up_ice(nCloudClass)     = RESHAPE ( (/ &
  !  delta_up_ice        !                                                delta_up_ice
   0.000_r8, &    ! classe  1:                                            0.962
   1.000_r8, &    ! classe  2:                                            0.961
   1.000_r8, &    ! classe  3:                                            0.966
   1.000_r8, &    ! classe  4:                                            0.966
   1.000_r8, &    ! classe  5:                                            0.965
   1.000_r8, &    ! classe  6:                                            0.960
   1.000_r8, &    ! classe  7:                                            0.950
   1.000_r8, &    ! classe  8:                                            0.960
   1.000_r8, &    ! classe  9:                                            0.962
   1.000_r8, &    ! classe 10:                                            0.952
   1.000_r8, &    ! classe 11:                                            0.970
   1.000_r8, &    ! classe 12:                                            0.950
   1.000_r8, &    ! classe 13:                                            0.914
   1.000_r8, &    ! classe 14:                                            0.970
   1.000_r8  &    ! classe 15:                                            0.970
   /), (/15/) )!---->  cloud systems  
 REAL(KIND=r8)    :: x(1:nCols),maxdepth(1:nCols)
 INTEGER :: ii,kk,i,j,k,inclass,lrec
 REAL(KIND=r8)    :: ice_ref   (1:nCols,1:nsoilay)
 REAL(KIND=r8)    :: liq_ref   (1:nCols,1:nsoilay)
 REAL(KIND=r8)    :: totdepth  (1:nCols)   ! total soil depth
 REAL(KIND=r8)    :: frootnorm1(1:nCols)   ! normalization factor for Jackson rooting profile,low
 REAL(KIND=r8)    :: frootnorm2(1:nCols)   ! normalization factor for Jackson rooting profile, up 
 REAL(KIND=r8)    :: depth     (1:nCols,1:nsoilay)
 REAL(KIND=r8)    :: frootm    (1:nCols,1:nsoilay,1:2)
 REAL(KIND=r8)    :: theta_lw_liq(1:nCols),theta_up_ice(1:nCols)
! ************************************************************************
! define class profiles
! ************************************************************************
!
! define class profiles based upon data published in:
!
! Jackson et al., 1996:  A global analysis of root distributions
! for terrestrial biomes, Oecologia, 108, 389-411.
!
! and
!
! Jackson et al., 1997:  A global budget for fine root biomass, 
! surface area, and nutrient contents, Proceedings of the National
! Academy of Sciences, 94, 7362-7366.
!
   inclass=1


      liq_ref(1:nCols,1)=1.0e-8_r8
      DO k = 2, nsoilay
         DO ii=1,nCols
            liq_ref(ii,k) = liq_ref(ii,k-1)  +  ((1e-05_r8 - 1.0e-10_r8)/nsoilay)
         END DO
      END DO

      ice_ref(1:nCols,1)=1.0e-10_r8
      DO k = 2, nsoilay
         DO ii=1,nCols
            ice_ref(ii,k) = ice_ref(ii,k-1)  +  ((1e-04_r8 - 1.0e-10_r8)/nsoilay)
         END DO
      END DO

      DO k = 1, nsoilay
         !
         IF (k.eq.1) THEN
            DO ii=1,nCols
               depth(ii,k) = liq_ref(ii,k)
            END DO
         ELSE
             DO ii=1,nCols
                depth(ii,k) = depth(ii,k-1) + liq_ref(ii,k)
             END DO
         END IF
         ! beta_lw_liq   o parametro de forma
         ! theta_lw_liq  o parametro de escala 
         ! delta_lw_liq  o parametro de localizao
         DO ii=1,nCols
            maxdepth(ii)= depth(ii,k)
         END DO
      END DO

   frcliq=0.0_r8
   DO kk=1,kMax
      frootnorm1(1:nCols)=0.0_r8
      DO k = 1, nsoilay
         !
         IF (k.eq.1) THEN
             DO ii=1,nCols
                depth(ii,k) = liq_ref(ii,k)
             END DO
         ELSE
             DO ii=1,nCols
                depth(ii,k) = depth(ii,k-1) + liq_ref(ii,k)
             END DO
         END IF
         ! beta_lw_liq   o parametro de forma
         ! theta_lw_liq  o parametro de escala 
         ! delta_lw_liq  o parametro de localizao
         DO ii=1,nCols
            theta_lw_liq(ii) = (1.1_r8 - theta_lw_liquid_RH(inclass))
            x(ii)=depth(ii,k)
            IF( x(ii) < delta_lw_liq (inclass))STOP 'ERROR depth < delta_lw_liq (inclass)'
            frootnorm1(ii) =frootnorm1(ii) +  (beta_lw_liq(inclass)/theta_lw_liq(ii))*(((x(ii)-delta_lw_liq(inclass))/theta_lw_liq(ii))**(beta_lw_liq(inclass)-1.0))*(exp(-(((x(ii)-delta_lw_liq(inclass))/theta_lw_liq(ii))**beta_lw_liq(inclass))))
         END DO
      END DO

      DO k = 1, nsoilay
      !
         IF (k.eq.1) THEN
             DO ii=1,nCols
                depth(ii,k) = liq_ref(ii,k)
             END DO
         ELSE
             DO ii=1,nCols
                depth(ii,k) = depth(ii,k-1) +liq_ref(ii,k)
             END DO
         END IF
         DO ii=1,nCols
            IF( depth(ii,k) <= gliq(ii,kk))THEN
               ! beta_lw_liq   o parametro de forma
               ! theta_lw_liq  o parametro de escala 
               ! delta_lw_liq  o parametro de localizao
               theta_lw_liq(ii) = (1.1_r8-(grh(ii,kk)))
               x(ii)=depth(ii,k)

               IF( x(ii) < delta_lw_liq (inclass))STOP 'ERROR depth < delta_lw_liq (inclass)'
               frootm(ii,k,1) = (beta_lw_liq(inclass)/theta_lw_liq(ii))*(((x(ii)-delta_lw_liq(inclass))/theta_lw_liq(ii))**(beta_lw_liq(inclass)-1.0))*(exp(-(((x(ii)-delta_lw_liq(inclass))/theta_lw_liq(ii))**beta_lw_liq(inclass))))

               frootm(ii,k,1) = frootm(ii,k,1) / frootnorm1(ii)
               frcliq(ii,kk)=frootm(ii,k,1)*100.0_r8
            END IF
         END DO
      END DO
   END DO


   frcice=0.0_r8
   DO kk=1,kMax
      frootnorm2(1:nCols)=0.0_r8
      DO k = 1, nsoilay
         !
         IF (k.eq.1) THEN
            DO ii=1,nCols
               depth(ii,k) = ice_ref(ii,k)
            END DO
         ELSE
            DO ii=1,nCols
              depth(ii,k) = depth(ii,k-1) + ice_ref(ii,k)
            END DO
         END IF

         ! beta_up_ice   o parametro de forma
         ! theta_up_ice  o parametro de escala 
         ! delta_up_ice  o parametro de localizao
         DO ii=1,nCols
            theta_up_ice(ii) = (1.1_r8-(theta_up_ice_RH(inclass) ))
            x(ii)=depth(ii,k)
            IF( x(ii) < delta_up_ice (inclass))STOP 'ERROR depth < delta_up_ice (inclass)'
            frootnorm2(ii) =frootnorm2(ii) +  (beta_up_ice(inclass)/theta_up_ice(ii))*(((x(ii)-delta_up_ice(inclass))/theta_up_ice(ii))**(beta_up_ice(inclass)-1.0_r8))*(exp(-(((x(ii)-delta_up_ice(inclass))/theta_up_ice(ii))**beta_up_ice(inclass))))
         END DO
      END DO

      DO k = 1, nsoilay
         !
         IF (k.eq.1) THEN
            DO ii=1,nCols
               depth(ii,k) = ice_ref(ii,k)
            END DO
         ELSE
            DO ii=1,nCols
               depth(ii,k) = depth(ii,k-1) +ice_ref(ii,k)
            END DO
         END IF
         DO ii=1,nCols
            IF( depth(ii,k) <= gice(ii,kk))THEN
               ! beta_up_ice   o parametro de forma
               ! theta_up_ice  o parametro de escala 
               ! delta_up_ice  o parametro de localizao
               theta_up_ice(ii) = (1.1_r8-(grh(ii,kk)))
               x(ii)            = depth(ii,k)
               IF( x(ii) < delta_up_ice (inclass))STOP 'ERROR depth < delta_up_ice (inclass)'
               frootm(ii,k,2) = (beta_up_ice(inclass)/theta_up_ice(ii))*(((x(ii)-delta_up_ice(inclass))/theta_up_ice(ii))**(beta_up_ice(inclass)-1.0_r8))*(exp(-(((x(ii)-delta_up_ice(inclass))/theta_up_ice(ii))**beta_up_ice(inclass))))

               frootm(ii,k,2) = frootm(ii,k,2) / frootnorm2(ii)
               frcice(ii,kk)  = frootm(ii,k,2)*1000.0_r8
            END IF
         END DO
!
      END DO
   END DO
!   DO k = 1, nsoilay
!      !
!      WRITE(*,*)'frcice=',frcice,'frcliq=',frcliq
!      
!    END DO
!      IF( x < 0) THEN
!          Fx(i,j) = 0.0_r8
!      ELSE
!          Fx(i,j) = 1 - exp(-(((x-delta)/theta)**(beta_lw_liq)))
!      END IF
 
!   DO k = 1, nsoilay
      !

!             PRINT*,'depth(k)=',depth(k),'frootm(i,k,1,j)=',frootm(k,1)


!   END DO
!
! return to main program
!
END SUBROUTINE CloudFractionLayrson

  REAL(KIND=r8) FUNCTION erf_cldf(x)
        ! # MS Fortran
        ! Error function from Numerical Recipes.
        ! erf(x) = 1 - erfc(x)

   IMPLICIT NONE

   REAL(KIND=r8) :: dumerfc, x
   REAL(KIND=r8) :: t, z


   z = abs(x)
   t = 1.0_r8 / ( 1.0_r8 + 0.5_r8 * z )

   dumerfc =       t * exp(-z * z - 1.26551223_r8 + t *          &
               ( 1.00002368_r8 + t * ( 0.37409196_r8 + t *            &
             ( 0.09678418_r8 + t * (-0.18628806_r8 + t *            &
                   ( 0.27886807_r8 + t * (-1.13520398_r8 + t *            &
             ( 1.48851587_r8 + t * (-0.82215223_r8 + t * 0.17087277_r8 )))))))))

   IF ( x.lt.0.0_r8 ) dumerfc = 2.0_r8 - dumerfc
     
   erf_cldf = 1.0_r8 - dumerfc

  END FUNCTION erf_cldf


END MODULE CloudFractionModel
