!
!  $Author: pkubota $
!  $Date: 2011/04/26 17:52:17 $
!  $Revision: 1.4 $
!  $Modified snilo 26/ag/2017. Excluded SW Scheme from COLA 
!
MODULE Rad_COLA

  ! InitRadiation
  !
  ! lwrad   ------| lwflux -----| crunch
  !               !
  !               ! cldslw

  USE Constants, ONLY :     &
       r8, i8

  IMPLICIT NONE

  PRIVATE

  PUBLIC :: InitRadCOLA
  PUBLIC :: lwrad

  ! Tabulated planck functions for LW
  REAL(KIND=r8) :: b2502 (32) ! water vapor
  REAL(KIND=r8) :: b2501 (32) ! water vapor
  REAL(KIND=r8) :: blkwin(32) ! ozone
  REAL(KIND=r8) :: blkco2(32) ! co2

  ! Water vapor absorption for SW
  INTEGER, PARAMETER :: nwaterbd = 10 ! number of sw water vapor bands
  REAL(KIND=r8)      :: xk(nwaterbd)  ! coefficients
  REAL(KIND=r8)      :: fk(nwaterbd)  ! weights

CONTAINS

  SUBROUTINE InitRadCOLA()

    ! planck function table for water vapor bands (center)
    b2501(:) = (/ &
         16.280e0_r8, 17.471e0_r8, 18.701e0_r8, 19.974e0_r8, 21.292e0_r8, &
         22.661e0_r8, 24.086e0_r8, 25.575e0_r8, 27.135e0_r8, 28.775e0_r8, &
         30.506e0_r8, 32.339e0_r8, 34.286e0_r8, 36.361e0_r8, 38.578e0_r8, &
         40.954e0_r8, 43.505e0_r8, 46.248e0_r8, 49.203e0_r8, 52.388e0_r8, &
         55.824e0_r8, 59.532e0_r8, 63.533e0_r8, 67.849e0_r8, 72.502e0_r8, &
         77.516e0_r8, 82.913e0_r8, 88.717e0_r8, 94.952e0_r8, 101.64e0_r8, &
         108.806e0_r8,116.472e0_r8 /)

    ! planck function table for water vapor bands (wing)
    b2502(:) = (/ &
         16.379e0_r8, 18.744e0_r8, 21.345e0_r8, 24.195e0_r8,  27.311e0_r8, &
         30.708e0_r8, 34.405e0_r8, 38.417e0_r8, 42.763e0_r8,  47.461e0_r8, &
         52.529e0_r8, 57.985e0_r8, 63.850e0_r8, 70.141e0_r8,  76.880e0_r8, &
         84.088e0_r8, 91.784e0_r8,  99.99e0_r8,108.726e0_r8, 118.016e0_r8, &
         127.881e0_r8,138.344e0_r8,149.429e0_r8,161.160e0_r8, 173.561e0_r8, &
         186.659e0_r8,200.478e0_r8,215.046e0_r8,230.390e0_r8, 246.539e0_r8, &
         263.523e0_r8,281.368e0_r8 /)

    ! planck function table for ozone band.
    blkwin(:) = (/ &
         0.593e0_r8, 0.774e0_r8, 0.993e0_r8, 1.258e0_r8, 1.573e0_r8, &
         1.944e0_r8, 2.377e0_r8, 2.877e0_r8, 3.450e0_r8, 4.102e0_r8, &
         4.838e0_r8, 5.664e0_r8, 6.585e0_r8, 7.606e0_r8, 8.733e0_r8, &
         9.969e0_r8,11.320e0_r8,12.788e0_r8,14.380e0_r8,16.097e0_r8, &
         17.944e0_r8,19.923e0_r8,22.038e0_r8,24.292e0_r8,26.685e0_r8, &
         29.221e0_r8,31.902e0_r8,34.729e0_r8,37.703e0_r8,40.825e0_r8, &
         44.097e0_r8,47.520e0_r8   /)

    ! planck function table for co2 bands.
    blkco2(:) = (/ &
         8.789e0_r8, 10.385e0_r8, 12.159e0_r8, 14.117e0_r8, 16.264e0_r8, &
         18.606e0_r8, 21.145e0_r8, 23.884e0_r8, 26.826e0_r8, 29.973e0_r8, &
         33.325e0_r8, 36.883e0_r8, 40.647e0_r8, 44.617e0_r8, 48.792e0_r8, &
         53.170e0_r8, 57.750e0_r8, 62.530e0_r8, 67.509e0_r8, 72.683e0_r8, &
         78.050e0_r8, 83.609e0_r8, 89.354e0_r8, 95.285e0_r8,101.397e0_r8, &
         107.688e0_r8,114.155e0_r8,120.794e0_r8,127.601e0_r8,134.574e0_r8, &
         141.710e0_r8,149.004e0_r8 /)


    ! Include global version 2.2 - Increase number of spectral radiation bands
    
    ! Ramaswamy &  Friedenreich data

    ! Water vapor absorption coefficients
    xk(:) = (/0.0002e1_r8,0.0035e1_r8,0.0377e1_r8,0.195e1_r8,0.940e1_r8, &
         4.46e1_r8,19.0e1_r8,98.9e1_r8,270.60e1_r8,3901.1e1_r8/)

    ! Water vapor absorption function weights
    fk(:) = (/0.0698e0_r8,0.1558e0_r8,0.0631e0_r8,0.0362e0_r8,0.0243e0_r8, &
         0.0158e0_r8,0.0087e0_r8,0.001467e0_r8,0.002342e0_r8,0.001075e0_r8/)

    ! Original COLA data
    !fk(:) = (/0.107e0_r8, 0.104e0_r8, 0.073e0_r8, 0.044e0_r8,  0.025e0_r8/)
    !xk(:) = (/0.005e0_r8, 0.041e0_r8, 0.416e0_r8, 4.752e0_r8, 72.459e0_r8/)

  END SUBROUTINE InitRadCOLA

  !------------------------------------------------------------------------
  ! LONG WAVE FLUXES CALCULATION
  !    Original Paper:
  !      Harshvardan et al, 1987: "A fast radiation parameterization for 
  !      atmospheric circulation models", J. Geophys. Res., v92, 1009-1016.
  !------------------------------------------------------------------------

  ! crunch: Computation of the gaseous transmission functions
  SUBROUTINE crunch(indx1 ,indx2 ,ncols ,kmax  ,h0p   ,h1p   ,ozone ,txuf  , &
       tv1   ,tv2   ,tui   ,tui2  ,x1    ,x2    ,cc    ,rawi  , &
       x3    ,x4    ,ch    ,css   ,ccu   ,shi   ,shu   ,wdel  , &
       fw    ,pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm , &
       vbarm ,wbarm ,fluxu ,fluxd )
    !
    !
    !  Input formal parameters :: index1, index2
    !  parameters used to calculate trnasmission functions
    !  From one level to another
    !==========================================================================
    ! ncols......Number of grid points on a gaussian latitude circle
    ! kmax......Number of grid points at vertical
    ! indx1.....parameters used to calculate trnasmission functions
    ! indx2.....parameters used to calculate trnasmission functions
    ! h0p.......constant h0p = 0.0e0 fac converts to degrees / time step
    ! h1p.......constant h1p = 1.0e0 fac converts to degrees / time step
    ! ozone.....set ozone logical variable  ozone = (.NOT. noz)
    ! txuf......1.used as matrix of g-functions for paths from each level
    !             to all other layers.
    !           2.used for transmission in co2 band.
    !           3.used for transmission in ozone band.
    !           4.in cldslw used for probability of clear line-of-sight
    !             from each level to all other layers for max overlap.
    ! tv1.......Working dimension
    ! tv2 ......Working dimension
    ! tui.......Working dimension
    ! tui2......Working dimension
    ! x1........path water vapor(e-type) and working dimension
    ! x2........path water vapor(band-center) and working dimension
    ! cc........planck function at level temperature for co2 bands.
    ! rawi......water vapor amount in layer.
    ! x3........path water vapor (band-wings) and working dimension
    ! x4........Working dimension
    ! ch........Probability of clear line-of-sight from level to top of
    !           the atmosphere.
    ! css.......Large scale cloud amount and working dimension
    ! ccu.......Cumulus cloud amount and working dimension
    ! shi.......Total transmission function (water vapor + CO2 + ozone)
    !           g-function for a path from level to top of atmosphere.
    ! shu.......Total transmission function (water vapor + CO2 + ozone)
    !           g-function for a path from level  of atmosphere to surface
    ! wdel......Ozone path, water vapor (e-TYPE) transmission function in
    !           9.6 mcm band
    ! fw........Ozone path multiplied bye pressure
    ! pai.......Pressure at middle of layer
    ! tai.......Temperature at middle of layer
    ! ozai......ozone amount in layer.
    ! ubar......scaled water vapor path length in window.
    ! vbar......scaled water vapor path length in center.
    ! wbar......scaled water vapor path length in wing.
    ! ubarm.....ubarm(i,2) = (ubar(i,2) + ubar(i,1)) * hp5
    ! vbarm.... planck function at level temperature for ozone band.
    ! wbarm.... ubarm(i,2) = (ubar(i,2) + ubar(i,1)) * hp5
    ! fluxu.....Ozone path
    ! fluxd.....Ozone path mutiplicated by pressure
    !
    !==========================================================================
    INTEGER, INTENT(IN   ) :: ncols
    INTEGER, INTENT(IN   ) :: kmax
    INTEGER, INTENT(IN   ) :: indx1
    INTEGER, INTENT(IN   ) :: indx2
    REAL(KIND=r8),    INTENT(IN   ) :: h0p
    REAL(KIND=r8),    INTENT(IN   ) :: h1p
    LOGICAL, INTENT(IN   ) :: ozone
    REAL(KIND=r8),    INTENT(INOUT  ) :: txuf  (ncols,kmax+2,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: tv1   (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(IN   ) :: tv2   (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(IN   ) :: tui   (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(IN   ) :: tui2  (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(INOUT  ) :: x1    (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(INOUT  ) :: x2    (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(IN   ) :: cc    (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(IN   ) :: rawi  (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(INOUT  ) :: x3    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT  ) :: x4    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ch    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: css   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ccu   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT  ) :: shi   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT  ) :: shu   (ncols,kmax+1)
    REAL(KIND=r8),    INTENT(INOUT  ) :: wdel  (ncols,kmax+1)
    REAL(KIND=r8),    INTENT(INOUT  ) :: fw    (ncols,kmax+1)

    ! Local Variables --->> Global Variables

    REAL(KIND=r8),    INTENT(IN   ) :: pai  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: tai  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ozai (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ubar (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: vbar (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: wbar (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ubarm(ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: vbarm(ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: wbarm(ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: fluxu(ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: fluxd(ncols,kmax+2)

    REAL(KIND=r8)                   :: adel  (ncols,kmax+1)
    REAL(KIND=r8)                   :: bdel  (ncols,kmax+1)
    REAL(KIND=r8)                   :: yv    (ncols,kmax+1)
    REAL(KIND=r8)                   :: zv    (ncols,kmax+1)
    REAL(KIND=r8)                   :: wv    (ncols,kmax+1)
    REAL(KIND=r8)                   :: fu    (ncols,kmax+1)
    REAL(KIND=r8)                   :: yw    (ncols,kmax+1)
    REAL(KIND=r8)                   :: zw    (ncols,kmax+1)
    REAL(KIND=r8)                   :: ww    (ncols,kmax+1)

    !  Local Parameter

    REAL(KIND=r8), PARAMETER        :: h9p8  = -9.79e0_r8
    REAL(KIND=r8), PARAMETER        :: h27p  = -27.0e0_r8
    REAL(KIND=r8), PARAMETER        :: hp61  = 6.15384615e-1_r8
    REAL(KIND=r8), PARAMETER        :: h15p1 = 15.1e0_r8
    REAL(KIND=r8), PARAMETER        :: h3p1  = -3.1e0_r8
    REAL(KIND=r8), PARAMETER        :: hp9   = 0.9e0_r8
    REAL(KIND=r8), PARAMETER        :: hp04  = -0.04e0_r8
    REAL(KIND=r8), PARAMETER        :: h16p  = 16.0e0_r8
    REAL(KIND=r8), PARAMETER        :: h6p7  = -6.7e0_r8
    REAL(KIND=r8), PARAMETER        :: h1013 = 1013.25e0_r8
    REAL(KIND=r8), PARAMETER        :: h1381 = 1381.12e0_r8
    REAL(KIND=r8), PARAMETER        :: hp88  = 0.8796e0_r8
    REAL(KIND=r8), PARAMETER        :: hp677 = 6.7675e-1_r8
    REAL(KIND=r8), PARAMETER        :: h4p4  = -4.398e0_r8
    REAL(KIND=r8), PARAMETER        :: hp38  = 3.84615384e-1_r8

    INTEGER                :: ip
    !INTEGER                :: ix
    INTEGER                :: i  ! loop indices
    INTEGER                :: k  ! loop indices
    !INTEGER                :: isub
    !INTEGER                :: ksub
    !INTEGER                :: imtn
    INTEGER                :: ip1
    INTEGER                ::  n
    INTEGER                :: i0
    INTEGER                :: i1
    INTEGER                :: i2

    REAL(KIND=r8)                   :: x1_s
    REAL(KIND=r8)                   :: x2_s
    REAL(KIND=r8)                   :: x3_s
    REAL(KIND=r8)                   :: adel_s
    REAL(KIND=r8)                   :: bdel_s
    REAL(KIND=r8)                   :: wdel_s
    REAL(KIND=r8)                   :: fw_s
    REAL(KIND=r8)                   :: fu_s
    REAL(KIND=r8)                   :: yw_s
    REAL(KIND=r8)                   :: ww_s
    REAL(KIND=r8)                   :: yv_s
    REAL(KIND=r8)                   :: wv_s
    REAL(KIND=r8)                   :: zv_s
    REAL(KIND=r8)                   :: zw_s

    IF (ozone) THEN
       IF (indx1 == indx2) THEN
          IF (indx2 == 1) THEN
             DO k = 1, kMax+1
                DO i = 1, ncols
                   x1(i,k) = ABS(ubar(i,k+1))
                   x2(i,k) = ABS(vbar(i,k+1))
                   x3(i,k) = ABS(wbar(i,k+1))
                   adel(i,k) = ABS(tai(i,k+1))
                   bdel(i,k) = ABS(ch(i,k+1))
                END DO
             END DO
          ELSE
             DO k = 1, kmax+1
                DO i = 1, ncols
                   x1(i,k)   = ubar(i,kmax+2) - ubar(i,k)
                   x2(i,k)   = vbar(i,kmax+2) - vbar(i,k)
                   x3(i,k)   = wbar(i,kmax+2) - wbar(i,k)
                   adel(i,k) = tai(i,kmax+2)  - tai(i,k)
                   bdel(i,k) = ch(i,kmax+2)   - ch(i,k)
                   wdel(i,k) = fluxu(i,kmax+2) - fluxu(i,k)
                   fw(i,k) = fluxd(i,kmax+2) - fluxd(i,k)
                   x1(i,k) = ABS(x1(i,k))
                   x2(i,k) = ABS(x2(i,k))
                   x3(i,k) = ABS(x3(i,k))
                   adel(i,k) = ABS(adel(i,k))
                   bdel(i,k) = ABS(bdel(i,k))
                   wdel(i,k) = ABS(wdel(i,k))
                   fw  (i,k) = ABS(fw  (i,k))
                END DO
             END DO
          END IF

          DO k=1,kmax+1
             DO i = 1, ncols
                fu_s = h9p8 * x1(i,k)
                yw_s = EXP(LOG(x1(i,k) + 1.0e-100_r8)*.83_r8)
                ww_s = EXP(LOG(x3(i,k) + 1.0e-100_r8)*.6_r8)
                ww_s = h1p + h16p  * ww_s
                ww_s = h6p7 * x3(i,k) / ww_s
                ww_s = h27p * yw_s + ww_s
                yw_s = EXP(LOG(adel(i,k) + 1.0e-100_r8)*.56_r8)
                yw_s = h1p  + h15p1     * yw_s
                yv_s = h3p1 * adel(i,k) / yw_s
                yv_s = yv_s       + ww_s
                yw_s = EXP(LOG(bdel(i,k) + 1.0e-100_r8)*.57_r8)
                yw_s = h1p  + hp9       * yw_s
                wv_s = hp04 * bdel(i,k) / yw_s
                wv_s = wv_s + ww_s
                fw_s   = fw(i,k)/ (wdel(i,k) * h1013)
                zv_s   = h1381 * wdel(i,k)  / (hp88 * fw_s)
                adel_s = h1p + zv_s
                adel_s = SQRT(adel_s)
                adel_s = h4p4 * fw_s * (adel_s - h1p)
                zv_s = EXP(adel_s)
                fw_s = h1p - hp677 * (h1p - zv_s)
                adel_s = EXP(yv_s)
                bdel_s = EXP(wv_s)
                wdel_s = EXP(fu_s)
                yw_s = SQRT(x2(i,k))
                zw_s = SQRT(x3(i,k))
                ww_s = ((h1p   + 32.2095e0_r8  * x1(i,k)) &
                     /  (h1p        + 52.85e0_r8      * x1(i,k)) &
                     +  (0.534874e0_r8 + 199.0e0_r8      * x1(i,k) &
                     -  1990.63e0_r8  * x1(i,k)      * x1(i,k)) &
                     *  zw_s &
                     / (h1p         + 333.244e0_r8    * x1(i,k))) &
                     / ((h1p        + 74.144e0_r8     * x1(i,k)) &
                     / (0.43368e0_r8   + 24.7442e0_r8    * x1(i,k)) &
                     * zw_s      + h1p)
                wv_s = (h1p   + 9.22411e0_r8  * yw_s &
                     + 33.1236e0_r8    * x2(i,k) &
                     + 176.396e0_r8    * x2(i,k)      * x2(i,k))
                wv_s = h1p   / wv_s
                ww_s = MAX(ww_s, h0p)
                wv_s = MAX(wv_s, h0p)
                x1_s = MIN(x1(i,k), 0.06e0_r8)
                x2_s = MIN(x2(i,k), 2.0e0_r8)
                x3_s = MIN(x3(i,k), 8.0e0_r8)
                yw_s = SQRT(x2_s)
                zw_s = SQRT(x3_s)
                fu_s = x1_s * x1_s
                yv_s = (0.0851069e0_r8 * yw_s        &
                     - 0.187096e0_r8   * x2_s  * yw_s &
                     + 0.323105e0_r8   * x2_s) * 0.1e0_r8
                zv_s = 0.239186e0_r8   * x2_s        &
                     - 0.0922289e0_r8  * x2_s  * yw_s &
                     - 0.0167413e0_r8  * x2_s  * x2_s
                zv_s = zv_s * 1.0e-3_r8
                yw_s = (5.6383e-4_r8    + 1.05173e0_r8  * x1_s &
                     - 39.0722e0_r8 * fu_s) &
                     / (h1p   + 202.357e0_r8  * x1_s) &
                     + (0.0779555e0_r8  + 4.40720e0_r8  * x1_s &
                     + 3.15851e0_r8 * fu_s)   * zw_s &
                     / (h1p   + 40.2298e0_r8  * x1_s) &
                     + (-0.0381305e0_r8 - 3.63684e0_r8  * x1_s &
                     + 7.98951e0_r8 * fu_s)   * x3_s &
                     / (h1p   + 62.5692e0_r8  * x1_s) &
                     + (6.21039e-3_r8 + 0.710061e0_r8 * x1_s &
                     - 2.85241e0_r8 * fu_s)   * x3_s &
                     / (h1p   + 70.2912e0_r8  * x1_s) &
                     * zw_s
                yw_s = 0.1e0_r8    * yw_s
                zw_s = (-2.99542e-4_r8 + 0.238219e0_r8 * x1_s &
                     + 0.519264e0_r8   * fu_s) &
                     / (h1p         + 10.7775e0_r8  * x1_s) &
                     + (-2.91325e-2_r8 - 2.30007e0_r8  * x1_s &
                     + 10.946e0_r8     * fu_s)   * zw_s &
                     / (h1p         + 63.519e0_r8   * x1_s) &
                     + (1.43812e-2_r8  + 1.80265e0_r8  * x1_s &
                     - 10.1311e0_r8    * fu_s)   * x3_s &
                     / (h1p         + 98.4758e0_r8  * x1_s) &
                     + (-2.39016e-3_r8 - 3.71427e-1_r8 * x1_s &
                     + 2.35443e0_r8    * fu_s)   * x3_s &
                     / (h1p         + 120.228e0_r8  * x1_s) &
                     * zw_s
                zw_s = 1.0e-3_r8   * zw_s
                adel_s = hp38 * adel_s + hp61 * bdel_s
                fw_s   = fw_s * wdel_s
                !
                yw(i,k) = yw_s
                ww(i,k) = ww_s
                yv(i,k) = yv_s
                wv(i,k) = wv_s
                fw(i,k) = fw_s
                zv(i,k) = zv_s
                adel(i,k)= adel_s
                zw(i,k) = zw_s
                x1(i,k) = x1_s
                x2(i,k) = x2_s
                x3(i,k) = x3_s
             END DO
          END DO

          IF (indx2 == 1) THEN
             DO k = 2, kmax+2
                DO i = 1, ncols
                   x1(i,k)   = wv(i,k-1) * tv1(i,1)
                   x2(i,k)   = yv(i,k-1) * tui(i,1) + h1p &
                        + zv(i,k-1)   * tui2(i,1)
                   x3(i,k)   = ww(i,k-1) * tv2(i,1)
                   x4(i,k)   = yw(i,k-1) * tui(i,1) + h1p &
                        + zw(i,k-1)   * tui2(i,1)
                   fw(i,k-1) = adel(i,k-1) * cc(i,1) &
                        + fw(i,k-1)   * rawi(i,1)
                END DO
             END DO

             DO k = 2, kmax+2
                DO i = 1, ncols
                   shi(i,k) = x1(i,k)*x2(i,k) + x3(i,k)*x4(i,k) + fw(i,k-1)
                END DO
             END DO

          ELSE

             DO k = 1, kmax+1
                DO i = 1, ncols
                   x1(i,k) =  wv(i,k) * tv1(i,(kmax+3)) &
                        * (yv(i,k) * tui(i,(kmax+3))  + h1p &
                        + zv(i,k)  * tui2(i,(kmax+3)))
                   x2(i,k) = ww(i,k)  * tv2(i,(kmax+3)) &
                        * (yw(i,k) * tui(i,(kmax+3))  + h1p &
                        +  zw(i,k) * tui2(i,(kmax+3)))
                   x3(i,k) =  wv(i,k) * tv1(i,(kmax+2)) &
                        * (yv(i,k) * tui(i,(kmax+2))    + h1p &
                        + zv(i,k)  * tui2(i,(kmax+2)))
                   x4(i,k) = ww(i,k)  * tv2(i,(kmax+2)) &
                        * (yw(i,k) * tui(i,(kmax+2))    + h1p &
                        +  zw(i,k) * tui2(i,(kmax+2)))
                   shu(i,k) = (cc(i,(kmax+3))-cc(i,(kmax+2)))*adel(i,k) &
                        + (rawi(i,(kmax+3)) - rawi(i,(kmax+2))) * fw(i,k)
                   shu(i,k) = x1(i,k) + x2(i,k) - x3(i,k) - x4(i,k) &
                        + shu(i,k)
                END DO
             END DO

          END IF
       ELSE
          IF (indx2 == (kmax+2)) THEN
             DO ip = indx1, indx2
                DO k = indx1, ip
                   DO i = 1, (ncols)
                      x1(i,k-1)   = ubar(i,ip) - ubarm(i,k)
                      x2(i,k-1)   = vbar(i,ip) - vbarm(i,k)
                      x3(i,k-1)   = wbar(i,ip) - wbarm(i,k)
                      adel(i,k-1) = tai(i,ip)  - css(i,k)
                      bdel(i,k-1) = ch(i,ip)   - ccu(i,k)
                      wdel(i,k-1) = fluxu(i,ip) - ozai(i,k)
                      fw(i,k-1)   = fluxd(i,ip) - pai(i,k)
                   END DO
                END DO
                DO k = 1, ip-1
                   DO i  = 1, ncols
                      x1_s = ABS(x1(i,k))
                      x2_s = ABS(x2(i,k))
                      x3_s = ABS(x3(i,k))
                      adel_s = ABS(adel(i,k))
                      bdel_s = ABS(bdel(i,k))
                      wdel_s = ABS(wdel(i,k))
                      fw_s = ABS(fw(i,k))
                      fu_s = h9p8 * x1_s
                      yw_s = EXP(LOG(x1_s + 1.0e-100_r8)*.83_r8)
                      ww_s = EXP(LOG(x3_s + 1.0e-100_r8)*.6_r8)
                      ww_s = h1p + h16p  * ww_s
                      ww_s = h6p7 * x3_s / ww_s
                      ww_s = h27p * yw_s + ww_s
                      yw_s = EXP(LOG(adel_s + 1.0e-100_r8)*.56_r8)
                      yw_s = h1p  + h15p1     * yw_s
                      yv_s = h3p1 * adel_s / yw_s
                      yv_s = yv_s       + ww_s
                      yw_s = EXP(LOG(bdel_s + 1.0e-100_r8)*.57_r8)
                      yw_s = h1p  + hp9       * yw_s
                      wv_s = hp04 * bdel_s / yw_s
                      wv_s = wv_s + ww_s
                      fw_s = fw_s / (wdel_s * h1013)
                      zv_s = h1381   * wdel_s  / (hp88 * fw_s)
                      adel_s = h1p   + zv_s
                      adel_s = SQRT(adel_s)
                      adel_s = h4p4 * fw_s * (adel_s - h1p)
                      zv_s = EXP(adel_s)
                      fw_s = h1p - hp677 * (h1p - zv_s)
                      adel_s = EXP(yv_s)
                      bdel_s = EXP(wv_s)
                      wdel_s = EXP(fu_s)
                      yw_s = SQRT(x2_s)
                      zw_s = SQRT(x3_s)
                      ww_s = ((h1p   + 32.2095e0_r8  * x1_s) &
                           /  (h1p        + 52.85e0_r8      * x1_s) &
                           +  (0.534874e0_r8 + 199.0e0_r8      * x1_s  &
                           -  1990.63e0_r8   * x1_s      * x1_s) &
                           *  zw_s     &
                           / (h1p         + 333.244e0_r8    * x1_s))&
                           / ((h1p        + 74.144e0_r8     * x1_s) &
                           / (0.43368e0_r8   + 24.7442e0_r8    * x1_s) &
                           * zw_s      + h1p)
                      wv_s = (h1p   + 9.22411e0_r8  * yw_s  &
                           + 33.1236e0_r8    * x2_s   &
                           + 176.396e0_r8    * x2_s      * x2_s)
                      wv_s = h1p   / wv_s
                      ww_s = MAX(ww_s, h0p)
                      wv_s = MAX(wv_s, h0p)
                      x1_s = MIN(x1_s, 0.06e0_r8)
                      x2_s = MIN(x2_s, 2.0e0_r8)
                      x3_s = MIN(x3_s, 8.0e0_r8)
                      yw_s = SQRT(x2_s)
                      zw_s = SQRT(x3_s)
                      fu_s = x1_s * x1_s
                      yv_s = (0.0851069e0_r8  * yw_s &
                           -  0.187096e0_r8 * x2_s  * yw_s &
                           +  0.323105e0_r8 * x2_s) * 0.1e0_r8
                      zv_s =  0.239186e0_r8   * x2_s &
                           -  0.0922289e0_r8  * x2_s  * yw_s &
                           -  0.0167413e0_r8  * x2_s  * x2_s
                      zv_s =  zv_s * 1.0e-3_r8
                      yw_s = (5.6383e-4_r8    + 1.05173e0_r8  * x1_s &
                           - 39.0722e0_r8 * fu_s) &
                           / (h1p   + 202.357e0_r8  * x1_s) &
                           + (0.0779555e0_r8  + 4.40720e0_r8  * x1_s  &
                           + 3.15851e0_r8 * fu_s)   * zw_s  &
                           / (h1p   + 40.2298e0_r8  * x1_s) &
                           + (-0.0381305e0_r8 - 3.63684e0_r8  * x1_s  &
                           + 7.98951e0_r8 * fu_s)   * x3_s  &
                           / (h1p   + 62.5692e0_r8  * x1_s) &
                           + (6.21039e-3_r8 + 0.710061e0_r8 * x1_s  &
                           - 2.85241e0_r8 * fu_s)   * x3_s  &
                           / (h1p   + 70.2912e0_r8  * x1_s) &
                           * zw_s
                      yw_s = 0.1e0_r8    * yw_s
                      zw_s = (-2.99542e-4_r8 + 0.238219e0_r8 * x1_s &
                           + 0.519264e0_r8   * fu_s) &
                           / (h1p         + 10.7775e0_r8  * x1_s) &
                           + (-2.91325e-2_r8 - 2.30007e0_r8  * x1_s &
                           + 10.946e0_r8     * fu_s)   * zw_s &
                           / (h1p         + 63.519e0_r8   * x1_s) &
                           + (1.43812e-2_r8  + 1.80265e0_r8  * x1_s &
                           - 10.1311e0_r8    * fu_s)   * x3_s &
                           / (h1p         + 98.4758e0_r8  * x1_s) &
                           + (-2.39016e-3_r8 - 3.71427e-1_r8 * x1_s &
                           + 2.35443e0_r8    * fu_s)   * x3_s &
                           / (h1p         + 120.228e0_r8  * x1_s) &
                           * zw_s
                      zw_s = 1.0e-3_r8   * zw_s
                      !
                      adel(i,k) = adel_s
                      bdel(i,k) = bdel_s
                      wdel(i,k) = wdel_s
                      fw(i,k)   = fw_s
                      !
                      ! fu(i,k) nao usa depois
                      yw(i,k)   = yw_s
                      ww(i,k)   = ww_s
                      yv(i,k)   = yv_s
                      wv(i,k)   = wv_s
                      zv(i,k)   = zv_s
                      zw(i,k)   = zw_s
                   END DO
                END DO
                DO k = 1, ip-1
                   DO i = 1, ncols
                      x1_s =  wv(i,k)  * tv1 (i,k) &
                           * (yv(i,k)  * tui (i,k)    + h1p &
                           +  zv(i,k)  * tui2(i,k))
                      x2_s =  ww(i,k)  * tv2 (i,k) &
                           * (yw(i,k)  * tui (i,k)    + h1p &
                           +  zw(i,k)  * tui2(i,k))
                      x3_s =  wv(i,k)  * tv1 (i,2+k-1) &
                           * (yv(i,k)  * tui (i,2+k-1) + h1p &
                           +  zv(i,k)  * tui2(i,2+k-1))
                      x4(i,k) =  ww(i,k)  * tv2 (i,2+k-1) &
                           * (yw(i,k)  * tui (i,2+k-1)   + h1p &
                           +  zw(i,k)  * tui2(i,2+k-1))
                      txuf(i,k,ip) =  x1_s  + x2_s &
                           -  x3_s      - x4(i,k) &
                           + (cc(i,1+k-1)    - cc(i,2+k-1)) &
                           * (hp38       * adel(i,k) &
                           +  hp61       * bdel(i,k)) &
                           + (rawi(i,1+k-1)  - rawi(i,2+k-1)) &
                           *  fw(i,k)      * wdel(i,k)
                      !
                      x1(i,k) = x1_s
                      x2(i,k) = x2_s
                      x3(i,k) = x3_s
                   END DO
                END DO
             END DO
          END IF


          IF (indx2 /= (kmax+2)) THEN
             DO ip = indx1, indx2
                DO k = indx1, (kmax+2)-ip
                   DO i = 1, (ncols)
                      x1(i,k-0)   = ubar(i,ip) - ubarm(i,k+ip)
                      x2(i,k-0)   = vbar(i,ip) - vbarm(i,k+ip)
                      x3(i,k-0)   = wbar(i,ip) - wbarm(i,k+ip)
                      adel(i,k-0) = tai(i,ip)  - css(i,k+ip)
                      bdel(i,k-0) = ch(i,ip) - ccu(i,k+ip)
                      wdel(i,k-0) = fluxu(i,ip) - ozai(i,k+ip)
                      fw(i,k-0)   = fluxd(i,ip) - pai(i,k+ip)
                   END DO
                END DO

                DO k = 1, (kmax+2)-ip
                   DO i  = 1, ncols
                      x1(i,k) = ABS(x1(i,k))
                      x2(i,k) = ABS(x2(i,k))
                      x3(i,k) = ABS(x3(i,k))
                      adel(i,k) = ABS(adel(i,k))
                      bdel(i,k) = ABS(bdel(i,k))
                      wdel(i,k) = ABS(wdel(i,k))
                      fw(i,k) = ABS(fw(i,k))
                      fu(i,k) = h9p8 * x1(i,k)
                      yw(i,k) = EXP(LOG(x1(i,k) + 1.0e-100_r8)*.83_r8)
                      ww(i,k) = EXP(LOG(x3(i,k) + 1.0e-100_r8)*.6_r8)
                      ww(i,k) = h1p + h16p  * ww(i,k)
                      ww(i,k) = h6p7 * x3(i,k) / ww(i,k)
                      ww(i,k) = h27p * yw(i,k) + ww(i,k)
                      yw(i,k) = EXP(LOG(adel(i,k) + 1.0e-100_r8)*.56_r8)
                      yw(i,k) = h1p  + h15p1  * yw(i,k)
                      yv(i,k) = h3p1 * adel(i,k) / yw(i,k)
                      yv(i,k) = yv(i,k)   + ww(i,k)
                      yw(i,k) = EXP(LOG(bdel(i,k) + 1.0e-100_r8)*.57_r8)
                      yw(i,k) = h1p  + hp9  * yw(i,k)
                      wv(i,k) = hp04 * bdel(i,k) / yw(i,k)
                      wv(i,k) = wv(i,k) + ww(i,k)
                      fw(i,k)    = fw(i,k) / (wdel(i,k) * h1013)
                      zv(i,k)    = h1381   * wdel(i,k)  / (hp88 * fw(i,k))
                      adel(i,k) = h1p      + zv(i,k)
                      adel(i,k) = SQRT(adel(i,k))
                      adel(i,k) = h4p4 * fw(i,k) * (adel(i,k) - h1p)
                      zv(i,k) = EXP(adel(i,k))
                      fw(i,k) = h1p - hp677 * (h1p - zv(i,k))
                      adel(i,k) = EXP(yv(i,k))
                      bdel(i,k) = EXP(wv(i,k))
                      wdel(i,k) = EXP(fu(i,k))
                      yw(i,k) = SQRT(x2(i,k))
                      zw(i,k) = SQRT(x3(i,k))
                      ww(i,k) = ((h1p      + 32.2095e0_r8    * x1(i,k)) &
                           /  (h1p   + 52.85e0_r8  * x1(i,k)) &
                           +  (0.534874e0_r8 + 199.0e0_r8  * x1(i,k)  &
                           -  1990.63e0_r8   * x1(i,k)  * x1(i,k)) &
                           *  zw(i,k)        &
                           / (h1p   + 333.244e0_r8  * x1(i,k)))&
                           / ((h1p   + 74.144e0_r8  * x1(i,k)) &
                           / (0.43368e0_r8   + 24.7442e0_r8  * x1(i,k)) &
                           * zw(i,k)   + h1p)
                      wv(i,k) = (h1p      + 9.22411e0_r8    * yw(i,k)  &
                           + 33.1236e0_r8    * x2(i,k)      &
                           + 176.396e0_r8    * x2(i,k)  * x2(i,k))
                      wv(i,k) = h1p      / wv(i,k)
                      ww(i,k) = MAX(ww(i,k), h0p)
                      wv(i,k) = MAX(wv(i,k), h0p)
                      x1(i,k) = MIN(x1(i,k), 0.06e0_r8)
                      x2(i,k) = MIN(x2(i,k), 2.0e0_r8)
                      x3(i,k) = MIN(x3(i,k), 8.0e0_r8)
                      yw(i,k) = SQRT(x2(i,k))
                      zw(i,k) = SQRT(x3(i,k))
                      fu(i,k) = x1(i,k) * x1(i,k)
                      yv(i,k) = (0.0851069e0_r8  * yw(i,k) &
                           -  0.187096e0_r8   * x2(i,k)  * yw(i,k) &
                           +  0.323105e0_r8   * x2(i,k)) * 0.1e0_r8
                      zv(i,k) =  0.239186e0_r8   * x2(i,k) &
                           -  0.0922289e0_r8  * x2(i,k)  * yw(i,k) &
                           -  0.0167413e0_r8  * x2(i,k)  * x2(i,k)
                      zv(i,k) =  zv(i,k) * 1.0e-3_r8

                      yw(i,k) = (5.6383e-4_r8    + 1.05173e0_r8  * x1(i,k) &
                           - 39.0722e0_r8     * fu(i,k)) &
                           / (h1p    + 202.357e0_r8  * x1(i,k)) &
                           + (0.0779555e0_r8  + 4.40720e0_r8  * x1(i,k)  &
                           + 3.15851e0_r8     * fu(i,k)) * zw(i,k)  &
                           / (h1p    + 40.2298e0_r8  * x1(i,k)) &
                           + (-0.0381305e0_r8 - 3.63684e0_r8  * x1(i,k)  &
                           + 7.98951e0_r8     * fu(i,k)) * x3(i,k)  &
                           / (h1p    + 62.5692e0_r8  * x1(i,k)) &
                           + (6.21039e-3_r8   + 0.710061e0_r8 * x1(i,k)  &
                           - 2.85241e0_r8     * fu(i,k)) * x3(i,k)  &
                           / (h1p    + 70.2912e0_r8  * x1(i,k)) &
                           * zw(i,k)
                      yw(i,k) = 0.1e0_r8       * yw(i,k)
                      zw(i,k) = (-2.99542e-4_r8 + 0.238219e0_r8 * x1(i,k) &
                           + 0.519264e0_r8   * fu(i,k)) &
                           / (h1p   + 10.7775e0_r8  * x1(i,k)) &
                           + (-2.91325e-2 - 2.30007e0_r8  * x1(i,k) &
                           + 10.946e0_r8   * fu(i,k))   * zw(i,k) &
                           / (h1p   + 63.519e0_r8   * x1(i,k)) &
                           + (1.43812e-2_r8  + 1.80265e0_r8  * x1(i,k) &
                           - 10.1311e0_r8    * fu(i,k))   * x3(i,k) &
                           / (h1p   + 98.4758e0_r8  * x1(i,k)) &
                           + (-2.39016e-3_r8 - 3.71427e-1_r8 * x1(i,k) &
                           + 2.35443e0_r8    * fu(i,k))   * x3(i,k) &
                           / (h1p   + 120.228e0_r8  * x1(i,k)) &
                           * zw(i,k)
                      zw(i,k) = 1.0e-3_r8       * zw(i,k)
                   END DO
                END DO

                DO k = 1, (kmax+2)-ip
                   DO i = 1, ncols
                      x1(i,k) =  wv(i,k)  * tv1 (i,ip+k) &
                           * (yv(i,k)  * tui (i,ip+k)  + h1p &
                           +  zv(i,k)  * tui2(i,ip+k))
                      x2(i,k) =  ww(i,k)  * tv2 (i,ip+k) &
                           * (yw(i,k)  * tui (i,ip+k)  + h1p &
                           +  zw(i,k)  * tui2(i,ip+k))
                      x3(i,k) =  wv(i,k)  * tv1 (i,ip+0+k-1) &
                           * (yv(i,k)  * tui (i,ip+0+k-1) + h1p &
                           +  zv(i,k)  * tui2(i,ip+0+k-1))
                      x4(i,k) =  ww(i,k)  * tv2 (i,ip+0+k-1) &
                           * (yw(i,k)  * tui (i,ip+0+k-1)   + h1p &
                           +  zw(i,k)  * tui2(i,ip+0+k-1))
                      txuf(i,ip+k,ip) =  x1(i,k)    + x2(i,k) &
                           -  x3(i,k)  - x4(i,k) &
                           + (cc(i,ip+1+k-1) - cc(i,ip+0+k-1)) &
                           * (hp38  * adel(i,k) &
                           +  hp61  * bdel(i,k)) &
                           + (rawi(i,ip+1+k-1)  - rawi(i,ip+0+k-1)) &
                           *  fw(i,k)  * wdel(i,k)
                   END DO
                END DO
             END DO
          END IF

       END IF
    ENDIF

    IF (.not.ozone) THEN
       IF (indx1 == indx2) THEN
          IF (indx2 == 1) THEN
             DO k = 1, kMax+1
                DO i = 1, ncols
                   x1(i,k) = ABS(ubar(i,k+1))
                   x2(i,k) = ABS(vbar(i,k+1))
                   x3(i,k) = ABS(wbar(i,k+1))
                   adel(i,k) = ABS(tai(i,k+1))
                   bdel(i,k) = ABS(ch(i,k+1))
                END DO
             END DO
          ELSE
             DO k = 1, kmax+1
                DO i = 1, ncols
                   x1(i,k)   = ubar(i,kmax+2) - ubar(i,k)
                   x2(i,k)   = vbar(i,kmax+2) - vbar(i,k)
                   x3(i,k)   = wbar(i,kmax+2) - wbar(i,k)
                   adel(i,k) = tai(i,kmax+2)  - tai(i,k)
                   bdel(i,k) = ch(i,kmax+2)   - ch(i,k)
                END DO
             END DO

             DO k = 1, kMax+1
                DO i = 1, ncols
                   x1(i,k) = ABS(x1(i,k))
                   x2(i,k) = ABS(x2(i,k))
                   x3(i,k) = ABS(x3(i,k))
                   adel(i,k) = ABS(adel(i,k))
                   bdel(i,k) = ABS(bdel(i,k))
                END DO
             END DO


          END IF
          DO k=1,(kmax+1)
             DO i = 1, (ncols)
                fu(i,k) = h9p8 * x1(i,k)
                yw(i,k) = EXP(LOG(x1(i,k) + 1.0e-100_r8)*.83_r8)
                ww(i,k) = EXP(LOG(x3(i,k) + 1.0e-100_r8)*.6_r8)
                ww(i,k) = h1p + h16p  * ww(i,k)
                ww(i,k) = h6p7 * x3(i,k) / ww(i,k)
                ww(i,k) = h27p * yw(i,k) + ww(i,k)
                yw(i,k) = EXP(LOG(adel(i,k) + 1.0e-100_r8)*.56_r8)
                yw(i,k) = h1p  + h15p1     * yw(i,k)
                yv(i,k) = h3p1 * adel(i,k) / yw(i,k)
                yv(i,k) = yv(i,k)       + ww(i,k)
                yw(i,k) = EXP(LOG(bdel(i,k) + 1.0e-100_r8)*.57_r8)
                yw(i,k) = h1p  + hp9       * yw(i,k)
                wv(i,k) = hp04 * bdel(i,k) / yw(i,k)
                wv(i,k) = wv(i,k) + ww(i,k)
             END DO
          END DO

          DO k=1,(kmax+1)
             DO i = 1, (ncols)
                fw(i,k) = h1p
             END DO
          END DO


          DO k = 1, kMax+1
             DO i = 1, ncols
                adel(i,k) = EXP(yv(i,k))
                bdel(i,k) = EXP(wv(i,k))
                wdel(i,k) = EXP(fu(i,k))
             END DO
          END DO

          DO k = 1, kMax+1
             DO i = 1, ncols
                yw(i,k) = SQRT(x2(i,k))
                zw(i,k) = SQRT(x3(i,k))
             END DO
          END DO

          DO k=1,(kmax+1)
             DO i = 1, (ncols)
                ww(i,k) = ((h1p   + 32.2095e0_r8  * x1(i,k)) &
                     /  (h1p        + 52.85e0_r8      * x1(i,k)) &
                     +  (0.534874e0_r8 + 199.0e0_r8      * x1(i,k) &
                     -  1990.63e0_r8   * x1(i,k)      * x1(i,k)) &
                     *  zw(i,k) &
                     / (h1p         + 333.244e0_r8    * x1(i,k))) &
                     / ((h1p        + 74.144e0_r8     * x1(i,k)) &
                     / (0.43368e0_r8   + 24.7442e0_r8    * x1(i,k)) &
                     * zw(i,k)      + h1p)
                wv(i,k) = (h1p   + 9.22411e0_r8  * yw(i,k) &
                     + 33.1236e0_r8    * x2(i,k) &
                     + 176.396e0_r8    * x2(i,k)      * x2(i,k))
                wv(i,k) = h1p   / wv(i,k)
             END DO
          END DO

          DO k = 1, kMax+1
             DO i = 1, ncols
                ww(i,k) = MAX(ww(i,k), h0p)
                wv(i,k) = MAX(wv(i,k), h0p)
                x1(i,k) = MIN(x1(i,k), 0.06e0_r8)
                x2(i,k) = MIN(x2(i,k), 2.0e0_r8)
                x3(i,k) = MIN(x3(i,k), 8.0e0_r8)
                yw(i,k) = SQRT(x2(i,k))
                zw(i,k) = SQRT(x3(i,k))
             END DO
          END DO
          DO k=1,(kmax+1)
             DO i = 1, (ncols)
                fu(i,k) = x1(i,k) * x1(i,k)
                yv(i,k) = (0.0851069e0_r8 * yw(i,k)        &
                     - 0.187096e0_r8   * x2(i,k)  * yw(i,k) &
                     + 0.323105e0_r8   * x2(i,k)) * 0.1e0_r8
                zv(i,k) = 0.239186e0_r8   * x2(i,k)        &
                     - 0.0922289e0_r8  * x2(i,k)  * yw(i,k) &
                     - 0.0167413e0_r8  * x2(i,k)  * x2(i,k)
                zv(i,k) = zv(i,k) * 1.0e-3_r8
                yw(i,k) = (5.6383e-4_r8    + 1.05173e0_r8  * x1(i,k) &
                     - 39.0722e0_r8 * fu(i,k)) &
                     / (h1p   + 202.357e0_r8  * x1(i,k)) &
                     + (0.0779555e0_r8  + 4.40720e0_r8  * x1(i,k) &
                     + 3.15851e0_r8 * fu(i,k))   * zw(i,k) &
                     / (h1p   + 40.2298e0_r8  * x1(i,k)) &
                     + (-0.0381305e0_r8 - 3.63684e0_r8  * x1(i,k) &
                     + 7.98951e0_r8 * fu(i,k))   * x3(i,k) &
                     / (h1p   + 62.5692e0_r8  * x1(i,k)) &
                     + (6.21039e-3_r8 + 0.710061e0_r8 * x1(i,k) &
                     - 2.85241e0_r8 * fu(i,k))   * x3(i,k) &
                     / (h1p   + 70.2912e0_r8  * x1(i,k)) &
                     * zw(i,k)
                yw(i,k) = 0.1e0_r8    * yw(i,k)
                zw(i,k) = (-2.99542e-4_r8 + 0.238219e0_r8 * x1(i,k) &
                     + 0.519264e0_r8   * fu(i,k)) &
                     / (h1p         + 10.7775e0_r8  * x1(i,k)) &
                     + (-2.91325e-2_r8 - 2.30007e0_r8  * x1(i,k) &
                     + 10.946e0_r8     * fu(i,k))   * zw(i,k) &
                     / (h1p         + 63.519e0_r8   * x1(i,k)) &
                     + (1.43812e-2_r8  + 1.80265e0_r8  * x1(i,k) &
                     - 10.1311e0_r8    * fu(i,k))   * x3(i,k) &
                     / (h1p         + 98.4758e0_r8  * x1(i,k)) &
                     + (-2.39016e-3_r8 - 3.71427e-1_r8 * x1(i,k) &
                     + 2.35443e0_r8    * fu(i,k))   * x3(i,k) &
                     / (h1p         + 120.228e0_r8  * x1(i,k)) &
                     * zw(i,k)
                zw(i,k) = 1.0e-3_r8   * zw(i,k)
             END DO
          END DO
          DO k=1,(kmax+1)
             DO i = 1, (ncols)
                adel(i,k) = hp38 * adel(i,k) + hp61 * bdel(i,k)
                fw(i,k)   = fw(i,k) * wdel(i,k)
             END DO
          END DO

          IF (indx2 == 1) THEN

             DO k = 2, (kmax+2)
                DO i = 1, (ncols)
                   x1(i,k)   = wv(i,k-1) * tv1(i,1)
                   x2(i,k)   = yv(i,k-1) * tui(i,1) + h1p &
                        + zv(i,k-1)   * tui2(i,1)
                   x3(i,k)   = ww(i,k-1) * tv2(i,1)
                   x4(i,k)   = yw(i,k-1) * tui(i,1) + h1p &
                        + zw(i,k-1)   * tui2(i,1)
                   fw(i,k-1) = adel(i,k-1) * cc(i,1) &
                        + fw(i,k-1)   * rawi(i,1)
                END DO
             END DO

             DO k = 2, (kmax+2)
                DO i = 1, (ncols)
                   shi(i,k) = x1(i,k)*x2(i,k) + x3(i,k)*x4(i,k) + fw(i,k-1)
                END DO
             END DO

          ELSE

             DO k = 1, (kmax+1)
                DO i = 1, (ncols)
                   x1(i,k) =  wv(i,k) * tv1(i,(kmax+3)) &
                        * (yv(i,k) * tui(i,(kmax+3))  + h1p &
                        + zv(i,k)  * tui2(i,(kmax+3)))
                   x2(i,k) = ww(i,k)  * tv2(i,(kmax+3)) &
                        * (yw(i,k) * tui(i,(kmax+3))  + h1p &
                        +  zw(i,k) * tui2(i,(kmax+3)))
                   x3(i,k) =  wv(i,k) * tv1(i,(kmax+2)) &
                        * (yv(i,k) * tui(i,(kmax+2))    + h1p &
                        + zv(i,k)  * tui2(i,(kmax+2)))
                   x4(i,k) = ww(i,k)  * tv2(i,(kmax+2)) &
                        * (yw(i,k) * tui(i,(kmax+2))    + h1p &
                        +  zw(i,k) * tui2(i,(kmax+2)))
                   shu(i,k) = (cc(i,(kmax+3))-cc(i,(kmax+2)))*adel(i,k) &
                        + (rawi(i,(kmax+3)) - rawi(i,(kmax+2))) * fw(i,k)
                END DO
             END DO
             DO k=1,(kmax+1)
                DO i = 1, (ncols)
                   shu(i,k) = x1(i,k) + x2(i,k) - x3(i,k) - x4(i,k) &
                        + shu(i,k)
                END DO
             END DO

          END IF
       ELSE
          DO ip = indx1, indx2

             IF (indx2 == (kmax+2)) THEN
                ip1 = ip
                n   = ip - 1
                i0  = 0
                i1  = 1
                i2  = 2
             ELSE
                ip1 = (kmax+2) - ip
                n   = ip1
                i0  = ip
                i1  = 0
                i2  = 0
             END IF

             DO k = indx1, ip1
                DO i = 1, (ncols)
                   x1(i,k-i1)   = ubar(i,ip) - ubarm(i,k+i0)
                   x2(i,k-i1)   = vbar(i,ip) - vbarm(i,k+i0)
                   x3(i,k-i1)   = wbar(i,ip) - wbarm(i,k+i0)
                   adel(i,k-i1) = tai(i,ip)  - css(i,k+i0)
                   bdel(i,k-i1) = ch(i,ip)   - ccu(i,k+i0)
                END DO
             END DO


             DO k = 1, n
                DO i  = 1, ncols
                   x1(i,k) = ABS(x1(i,k))
                   x2(i,k) = ABS(x2(i,k))
                   x3(i,k) = ABS(x3(i,k))
                   adel(i,k) = ABS(adel(i,k))
                   bdel(i,k) = ABS(bdel(i,k))
                END DO
             END DO

             DO k= 1, n
                DO i= 1, ncols
                   fu(i,k) = h9p8 * x1(i,k)
                   yw(i,k) = EXP(LOG(x1(i,k) + 1.0e-100_r8)*.83_r8)
                   ww(i,k) = EXP(LOG(x3(i,k) + 1.0e-100_r8)*.6_r8)
                   ww(i,k) = h1p + h16p  * ww(i,k)
                   ww(i,k) = h6p7 * x3(i,k) / ww(i,k)
                   ww(i,k) = h27p * yw(i,k) + ww(i,k)
                   yw(i,k) = EXP(LOG(adel(i,k) + 1.0e-100_r8)*.56_r8)
                   yw(i,k) = h1p  + h15p1     * yw(i,k)
                   yv(i,k) = h3p1 * adel(i,k) / yw(i,k)
                   yv(i,k) = yv(i,k)       + ww(i,k)
                   yw(i,k) = EXP(LOG(bdel(i,k) + 1.0e-100_r8)*.57_r8)
                   yw(i,k) = h1p  + hp9       * yw(i,k)
                   wv(i,k) = hp04 * bdel(i,k) / yw(i,k)
                   wv(i,k) = wv(i,k) + ww(i,k)
                END DO
             END DO


             DO k=1, n
                DO i = 1, ncols
                   fw(i,k) = h1p
                END DO
             END DO

             DO k = 1, n
                DO i = 1, ncols
                   adel(i,k) = EXP(yv(i,k))
                   bdel(i,k) = EXP(wv(i,k))
                   wdel(i,k) = EXP(fu(i,k))
                END DO
             END DO
             DO k = 1, n
                DO i = 1, ncols
                   yw(i,k) = SQRT(x2(i,k))
                   zw(i,k) = SQRT(x3(i,k))
                END DO
             END DO

             DO k = 1, n
                DO i = 1, ncols
                   ww(i,k) = ((h1p   + 32.2095e0_r8  * x1(i,k)) &
                        /  (h1p        + 52.85e0_r8      * x1(i,k)) &
                        +  (0.534874e0_r8 + 199.0e0_r8      * x1(i,k)  &
                        -  1990.63e0_r8   * x1(i,k)      * x1(i,k)) &
                        *  zw(i,k)     &
                        / (h1p         + 333.244e0_r8    * x1(i,k)))&
                        / ((h1p        + 74.144e0_r8     * x1(i,k)) &
                        / (0.43368e0_r8   + 24.7442e0_r8    * x1(i,k)) &
                        * zw(i,k)      + h1p)
                   wv(i,k) = (h1p   + 9.22411e0_r8  * yw(i,k)  &
                        + 33.1236e0_r8    * x2(i,k)   &
                        + 176.396e0_r8    * x2(i,k)      * x2(i,k))
                   wv(i,k) = h1p   / wv(i,k)
                END DO
             END DO

             DO k = 1, n
                DO i = 1, ncols
                   ww(i,k) = MAX(ww(i,k), h0p)
                   wv(i,k) = MAX(wv(i,k), h0p)
                   x1(i,k) = MIN(x1(i,k), 0.06e0_r8)
                   x2(i,k) = MIN(x2(i,k), 2.0e0_r8)
                   x3(i,k) = MIN(x3(i,k), 8.0e0_r8)
                   yw(i,k) = SQRT(x2(i,k))
                   zw(i,k) = SQRT(x3(i,k))
                END DO
             END DO

             DO k =1, n
                DO i = 1, ncols
                   fu(i,k) = x1(i,k) * x1(i,k)
                   yv(i,k) = (0.0851069e0_r8  * yw(i,k) &
                        -  0.187096e0_r8 * x2(i,k)  * yw(i,k) &
                        +  0.323105e0_r8 * x2(i,k)) * 0.1e0_r8
                   zv(i,k) =  0.239186e0_r8   * x2(i,k) &
                        -  0.0922289e0_r8  * x2(i,k)  * yw(i,k) &
                        -  0.0167413e0_r8  * x2(i,k)  * x2(i,k)
                   zv(i,k) =  zv(i,k) * 1.0e-3_r8

                   yw(i,k) = (5.6383e-4_r8    + 1.05173e0_r8  * x1(i,k) &
                        - 39.0722e0_r8 * fu(i,k)) &
                        / (h1p   + 202.357e0_r8  * x1(i,k)) &
                        + (0.0779555e0_r8  + 4.40720e0_r8  * x1(i,k)  &
                        + 3.15851e0_r8 * fu(i,k))   * zw(i,k)  &
                        / (h1p   + 40.2298e0_r8  * x1(i,k)) &
                        + (-0.0381305e0_r8 - 3.63684e0_r8  * x1(i,k)  &
                        + 7.98951e0_r8 * fu(i,k))   * x3(i,k)  &
                        / (h1p   + 62.5692e0_r8  * x1(i,k)) &
                        + (6.21039e-3_r8 + 0.710061e0_r8 * x1(i,k)  &
                        - 2.85241e0_r8 * fu(i,k))   * x3(i,k)  &
                        / (h1p   + 70.2912e0_r8  * x1(i,k)) &
                        * zw(i,k)
                   yw(i,k) = 0.1e0_r8    * yw(i,k)
                   zw(i,k) = (-2.99542e-4_r8 + 0.238219e0_r8 * x1(i,k) &
                        + 0.519264e0_r8   * fu(i,k)) &
                        / (h1p         + 10.7775e0_r8  * x1(i,k)) &
                        + (-2.91325e-2_r8 - 2.30007e0_r8  * x1(i,k) &
                        + 10.946e0_r8     * fu(i,k))   * zw(i,k) &
                        / (h1p         + 63.519e0_r8   * x1(i,k)) &
                        + (1.43812e-2_r8  + 1.80265e0_r8  * x1(i,k) &
                        - 10.1311e0_r8    * fu(i,k))   * x3(i,k) &
                        / (h1p         + 98.4758e0_r8  * x1(i,k)) &
                        + (-2.39016e-3_r8 - 3.71427e-1_r8 * x1(i,k) &
                        + 2.35443e0_r8    * fu(i,k))   * x3(i,k) &
                        / (h1p         + 120.228e0_r8  * x1(i,k)) &
                        * zw(i,k)
                   zw(i,k) = 1.0e-3_r8   * zw(i,k)
                END DO
             END DO

             DO k = 1, n
                DO i = 1, ncols
                   x1(i,k) =  wv(i,k)  * tv1 (i,i0+k) &
                        * (yv(i,k)  * tui (i,i0+k)    + h1p &
                        +  zv(i,k)  * tui2(i,i0+k))
                   x2(i,k) =  ww(i,k)  * tv2 (i,i0+k) &
                        * (yw(i,k)  * tui (i,i0+k)    + h1p &
                        +  zw(i,k)  * tui2(i,i0+k))
                   x3(i,k) =  wv(i,k)  * tv1 (i,i0+i2+k-1) &
                        * (yv(i,k)  * tui (i,i0+i2+k-1) + h1p &
                        +  zv(i,k)  * tui2(i,i0+i2+k-1))
                   x4(i,k) =  ww(i,k)  * tv2 (i,i0+i2+k-1) &
                        * (yw(i,k)  * tui (i,i0+i2+k-1)   + h1p &
                        +  zw(i,k)  * tui2(i,i0+i2+k-1))
                   txuf(i,i0+k,ip) =  x1(i,k)  + x2(i,k) &
                        -  x3(i,k)      - x4(i,k) &
                        + (cc(i,i0+1+k-1)    - cc(i,i0+i2+k-1)) &
                        * (hp38       * adel(i,k) &
                        +  hp61       * bdel(i,k)) &
                        + (rawi(i,i0+1+k-1)  - rawi(i,i0+i2+k-1)) &
                        *  fw(i,k)      * wdel(i,k)
                END DO
             END DO
          END DO
       END IF
    END IF
  END SUBROUTINE crunch


  ! lwflux :computes carbon dioxide and ozone fluxes and upward and downward
  !         fluxes txuf for water vapor; the transmittances are also
  !         calculated.
  SUBROUTINE lwflux(pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm ,vbarm , &
       wbarm ,fluxu ,fluxd ,txuf  ,tv1   ,tv2   ,tui   ,x1    , &
       x2    ,cc    ,rawi  ,x3    ,x4    ,ch    ,dp    ,css   , &
       ccu   ,shi   ,shh   ,shu   ,sumsav,h0p   ,h1p   ,h1p5  , &
       hp5   ,dtb   ,dtbinv,pr    ,ntm1  ,ozone ,co2m, &
       ncols ,kmax  )
    !
    !==========================================================================
    !
    !   ncols......Number of grid points on a gaussian latitude circle
    !   kmax......Number of grid points at vertical
    !   co2m....co2val is wgne standard value in ppm "co2val = /345.0/
    !   h0p.......constant h0p = 0.0e0 fac converts to degrees / time step
    !   h1p.......constant h1p = 1.0e0 fac converts to degrees / time step
    !   h1p5......Fact converts absorption to rate in degrees/ time step
    !             constant h1p5   = 1.5e0
    !   hp5.......constant hp5    = 0.5e0
    !   dtb.......temperature increment in b250.  Constant dtb  = 5.0e0
    !   dtbinv....constant dtbinv = h1p / dtb
    !   pr(1).....constant pr(1)  = h1p / 3.0e01
    !   pr(1).....constant pr(2)  = h1p / 3.0e02
    !   ntm1......number of rows in b250 - 1. constant  ntm1   = 31
    !   ozone.....set ozone logical variable  ozone = (.NOT. noz)
    !             true if there is ozone  absorption computation
    !   txuf......1.used as matrix of g-functions for paths from each level
    !               to all other layers.
    !             2.used for transmission in co2 band.
    !             3.used for transmission in ozone band.
    !             4.in cldslw used for probability of clear line-of-sight
    !               from each level to all other layers for max overlap.
    !   tv1.......Working dimension
    !   tv2 ......Working dimension
    !   tui.......Working dimension
    !   x1........path water vapor(e-type) and working dimension
    !   x2........path water vapor(band-center) and working dimension
    !   cc........planck function at level temperature for co2 bands.
    !   rawi......water vapor amount in layer
    !   x3........path water vapor (band-wings) and working dimension
    !   x4........Working dimension
    !   ch........Probability of clear line-of-sight from level to top of
    !             the atmosphere.
    !   dp........Pressure difference between levels
    !   css.......Large scale cloud amount and working dimension
    !   ccu.......Cumulus cloud amount and working dimension
    !   shi.......Total transmission function (water vapor + CO2 + ozone)
    !             g-function for a path from level to top of atmosphere.
    !   shh.......planck function at level temperature for water vapor
    !              bands.
    !   shu.......Total transmission function (water vapor + CO2 + ozone)
    !             g-function for a path from level  of atmosphere to surface
    !   sumsav....
    !   pai.......Pressure at middle of layer
    !   tai.......Temperature at middle of layer
    !   ozai......ozone amount in layer.
    !   ubar......scaled water vapor path length in window.
    !   vbar......scaled water vapor path length in center.
    !   wbar......scaled water vapor path length in wing.
    !   ubarm.....ubarm(i,2) = (ubar(i,2) + ubar(i,1)) * hp5
    !   vbarm.... planck function at level temperature for ozone band.
    !   wbarm.... ubarm(i,2) = (ubar(i,2) + ubar(i,1)) * hp5
    !   fluxu.....Ozone path
    !   fluxd.....Ozone path mutiplicated by pressure
    !
    !==========================================================================
    INTEGER, INTENT(in   ) :: ncols
    INTEGER, INTENT(in   ) :: kmax
    REAL(KIND=r8),    INTENT(in   ) :: h0p
    REAL(KIND=r8),    INTENT(in   ) :: h1p
    REAL(KIND=r8),    INTENT(in   ) :: h1p5
    REAL(KIND=r8),    INTENT(in   ) :: hp5
    REAL(KIND=r8),    INTENT(in   ) :: dtb
    REAL(KIND=r8),    INTENT(in   ) :: dtbinv
    REAL(KIND=r8),    INTENT(in   ) :: pr(2)
    INTEGER, INTENT(in   ) :: ntm1
    LOGICAL, INTENT(in   ) :: ozone
    REAL(KIND=r8),    INTENT(inout  ) :: txuf  (ncols,kmax+2,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: tv1   (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: tv2   (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout) :: tui   (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: x1    (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: x2    (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: cc    (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: rawi  (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: x3    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: x4    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: ch    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(in   ) :: dp    (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: css   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: ccu   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: shi   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: shh   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: shu   (ncols,kmax+1)
    REAL(KIND=r8),    INTENT(inout  ) :: sumsav(ncols)

    REAL(KIND=r8),    INTENT(INOUT) :: pai   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT) :: tai   (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT) :: ozai  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ubar  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: vbar  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: wbar  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: ubarm (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: vbarm (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(IN   ) :: wbarm (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT) :: fluxu (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(INOUT) :: fluxd (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(in   ) :: co2m  (nCols,kMax) !mol/mol

    REAL(KIND=r8)                   :: auxco2  (ncols,kmax+2)
    REAL(KIND=r8)                   :: tui2  (ncols,kmax+3)
    REAL(KIND=r8)                   :: wdel  (ncols,kmax+1)
    REAL(KIND=r8)                   :: fw   (ncols,kmax+1)
    INTEGER                :: it    (ncols,kmax+3)


    !
    !     REAL(KIND=r8) constants
    !
    REAL(KIND=r8), PARAMETER :: hp25=0.26_r8
    REAL(KIND=r8), PARAMETER :: temp1=165.0_r8
    REAL(KIND=r8), PARAMETER :: h250p=250.0_r8
    REAL(KIND=r8), PARAMETER :: pp=1.0_r8

    INTEGER :: i
    INTEGER :: j
    INTEGER :: k
    INTEGER :: ix
    INTEGER :: ip
    INTEGER :: indx1
    INTEGER :: indx2

    DO i = 1, ncols
       shi(i,1) = h0p
    END DO
    DO k=1,kmax
       DO i = 1, ncols
          auxco2(i,k) = co2m(i,k)*1e6_r8 !convet mol/mol to ppm
       END DO
    END DO
    DO i = 1, ncols
       auxco2(i,kmax+1) = co2m(i,kmax)*1e6_r8!convet mol/mol to ppm
       auxco2(i,kmax+2) = co2m(i,kmax)*1e6_r8!convet mol/mol to ppm
   END DO

    DO k=1,(kmax+2)
       DO j=1,(kmax+2)
          DO i = 1, ncols
             txuf(i,j,k) = h0p
          END DO
       END DO
    END DO

    DO k=1,kmax+3
       DO i = 1, ncols
          rawi(i,k) = (tui(i,k) - temp1) * dtbinv + h1p5
          it(i,k)   = rawi(i,k)
       END DO
    END DO

    DO k = 1, kmax+3
       DO i = 1, ncols
          it(i,k) = MAX(1,MIN(it(i,k), ntm1))
       END DO
    END DO

    DO k=1,(kmax+3)
       DO i = 1, (ncols)
          rawi(i,k) = it  (i,k)  -  h1p
          rawi(i,k) = tui (i,k)  - (temp1 + rawi(i,k) * dtb)
          rawi(i,k) = rawi(i,k)  *  dtbinv
       END DO
    END DO

    DO k = 1, kmax+3
       DO i = 1, ncols
          IF(it(i,k) .LE. ntm1 )THEN
             x1(i,k)=b2501(it(i,k))
             x2(i,k)=b2501(it(i,k)+1)
          ENDIF
       END DO
    END DO

    DO k=1,(kmax+3)
       DO i = 1, (ncols)
          tv1(i,k) = x1(i,k) + (x2(i,k) - x1(i,k)) * rawi(i,k)
       END DO
    END DO

    DO k = 1, kmax+3
       DO i = 1, ncols
          IF(it(i,k) .LE. ntm1 )THEN
             x2(i,k)=b2502(it(i,k)+1)
             x1(i,k)=b2502(it(i,k))
          ENDIF
       END DO
    END DO

    DO k=1,(kmax+3)
       DO i = 1, (ncols)
          tv2(i,k) = x1(i,k) + (x2(i,k) - x1(i,k)) * rawi(i,k)
       END DO
    END DO

    DO k=1,(kmax+2)
       DO i = 1, (ncols)
          shh(i,k) = tv1(i,k) + tv2(i,k)
       END DO
    END DO

    DO i = 1, ncols
       sumsav(i) = tv1(i,(kmax+3)) + tv2(i,(kmax+3))
    END DO

    DO k=1,(kmax+3)
       DO i = 1, (ncols)
          tui (i,k)  = tui(i,k) - h250p
          tui2(i,k)  = tui(i,k) * tui(i,k)
       END DO
    END DO
    !
    !     carbon dioxide and ozone fluxes are calculated here.
    !
    DO k = 1, kmax+3
       DO i = 1, ncols
          IF(it(i,k) .LE. ntm1 )THEN
             x1(i,k)=blkco2(it(i,k)+1)
             x2(i,k)=blkco2(it(i,k))
          ENDIF
       END DO
    END DO

    DO k=1,(kmax+3)
       DO i = 1, (ncols)
          cc(i,k) = x2(i,k) + (x1(i,k) - x2(i,k)) * rawi(i,k)
       ENDDO
    END DO

    DO k = 1, kmax+3
       DO i = 1, ncols
          IF(it(i,k) .LE. ntm1 )THEN
             x2(i,k)=blkwin(it(i,k))
             x1(i,k)=blkwin(it(i,k)+1)
          ENDIF
       END DO
    END DO

    DO k=1,(kmax+3)
       DO i = 1, (ncols)
          rawi(i,k) = x2(i,k) + (x1(i,k) - x2(i,k)) * rawi(i,k)
       END DO
    END DO
    !
    !     compute transmittances in the 15 micron and 9.6 micron bands.
    !
    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          x1(i,k) = 2.5e-2_r8 * (tai(i,k+1) - 240.0e0_r8)
       END DO
    END DO

    ch(1:ncols,2:(kmax+2))=exp(x1(1:ncols,1:(kmax+1)))

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          x1(i,k) = 8.9e-3_r8 * (tai(i,k+1) - 240.0e0_r8)
       END DO
    END DO

    tai(1:ncols,2:(kmax+2))=exp(x1(1:ncols,1:(kmax+1)))

    DO i = 1, ncols
       tai(i,1) = h0p
       ch(i,1)  = h0p
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          x1(i,k) = pai(i,k)
       END DO
    END DO

    DO k=1,kmax+1
       DO i = 1, ncols
          x1(i,k)=MAX(pp,x1(i,k))
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          fw(i,k) = x1(i,k) * pr(1)
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          x2(i,k) = EXP(0.85_r8* LOG(fw(i,k)))
       ENDDO
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          tai(i,k+1) = x2(i,k) * tai(i,k+1)
          x1(i,k)  = x1(i,k) * pr(2)
       END DO
    END DO

    x1(1:ncols,1:kmax+1) = sqrt(x1(1:ncols,1:kmax+1))

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          ch(i,k+1) = x1(i,k) * ch(i,k+1)
          !
          !     hp25 = 0.26 for co2 = 330 ppmv
          !
          tai(i,k+1) = dp(i,k+1) * tai(i,k+1) * hp25 * auxco2(i,k) / 330.0_r8
          ch(i,k+1)  = dp(i,k+1) * ch (i,k+1) * hp25 * auxco2(i,k) / 330.0_r8
       END DO
    END DO

    DO ip = 2, (kmax+2)
       DO i = 1, ncols
          tai(i,ip) = tai(i,ip-1) + tai(i,ip)
          ch(i,ip)  = ch(i,ip-1)  + ch(i,ip)
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          css(i,k+1) = (tai(i,k+1)  + tai(i,k))  * hp5
          ccu(i,k+1) = ( ch(i,k+1)  +  ch(i,k))  * hp5
       END DO
    END DO

    IF ( ozone ) THEN

       DO i = 1, ncols
          fluxd(i,1) = h0p
       END DO

       DO k=1,(kmax+2)
          DO i = 1, (ncols)
             fluxu(i,k) = h0p
          END DO
       END DO

       DO k=1,(kmax+1)
          DO i = 1, (ncols)
             fluxd(i,k+1) = pai(i,k) * ozai(i,k+1)
          END DO
       END DO

       DO ix = 2, (kmax+2)
          DO i = 1, ncols
             fluxd(i,ix) = fluxd(i,ix-1) + fluxd(i,ix)
             fluxu(i,ix) = fluxu(i,ix-1) + ozai(i,ix)
          END DO
       END DO

       DO k = 1, (kmax+1)
          DO i = 1, (ncols)
             pai (i,k+1)  = (fluxd(i,k) + fluxd(i,k+1)) * hp5
             ozai(i,k+1)  = (fluxu(i,k) + fluxu(i,k+1)) * hp5
          ENDDO
       END DO

       DO k=1,kmax+1
          DO i=1, ncols
             wdel(i,k)=ABS(fluxu(i,k+1))
             fw(i,k)= ABS(fluxd(i,k+1))
          END DO
       END DO
    END IF
    indx1 = 1
    indx2 = 1
    CALL crunch(indx1 ,indx2 ,ncols  ,kmax  ,h0p   ,h1p   ,ozone ,txuf  , &
         tv1   ,tv2   ,tui   ,tui2  ,x1    ,x2    ,cc    ,rawi  , &
         x3    ,x4    ,ch    ,css   ,ccu   ,shi   ,shu   ,wdel  , &
         fw    ,pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm , &
         vbarm ,wbarm ,fluxu ,fluxd )
    indx1 = 2
    indx2 = 2
    CALL crunch(indx1 ,indx2 ,ncols ,kmax  ,h0p   ,h1p   ,ozone ,txuf  , &
         tv1   ,tv2   ,tui   ,tui2  ,x1    ,x2    ,cc    ,rawi  , &
         x3    ,x4    ,ch    ,css   ,ccu   ,shi   ,shu   ,wdel  , &
         fw    ,pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm , &
         vbarm ,wbarm ,fluxu ,fluxd )

    !
    !     downward flux txuf for water vapor
    !

    indx1 = 2
    indx2 = (kmax+2)
    CALL crunch(indx1 ,indx2 ,ncols ,kmax  ,h0p   ,h1p   ,ozone ,txuf  , &
         tv1   ,tv2   ,tui   ,tui2  ,x1    ,x2    ,cc    ,rawi  , &
         x3    ,x4    ,ch    ,css   ,ccu   ,shi   ,shu   ,wdel  , &
         fw    ,pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm , &
         vbarm ,wbarm ,fluxu ,fluxd )
    !
    !     upward flux txuf for water vapor
    !
    indx1 = 1
    indx2 = (kmax+1)

    CALL crunch(indx1 ,indx2 ,ncols  ,kmax  ,h0p   ,h1p   ,ozone ,txuf  , &
         tv1   ,tv2   ,tui   ,tui2  ,x1    ,x2    ,cc    ,rawi  , &
         x3    ,x4    ,ch    ,css   ,ccu   ,shi   ,shu   ,wdel  , &
         fw    ,pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm , &
         vbarm ,wbarm ,fluxu ,fluxd )

    DO k = 1,(kmax+2)
       DO i = 1, (ncols)
          shh(i,k) = shh(i,k) + rawi(i,k) + cc(i,k)
       END DO
    END DO

    DO i = 1, ncols
       sumsav(i) = sumsav(i) + rawi(i,(kmax+3)) + cc(i,(kmax+3))
    END DO
  END SUBROUTINE lwflux



  !cldslw:  Estimate a contribution from cloudiness
  !    Using cloud amount of large scale(css) and cumulus(ccu) clouds
  SUBROUTINE cldslw(ncols ,kmax  ,nlcs  ,h1p   ,cs    ,x1    , &
       x2    ,cc    ,x3    ,x4    ,ch    ,css   ,ccu)
    !
    !==========================================================================
    !  imax.......Number of grid points on a gaussian latitude circle
    !  kmax.......Number of grid points at vertical
    !  nlcs.......nlcs=30
    !  h1p........h1p    = 1.0e0      fac converts to degrees / time step
    !  cs.........probability of clear line-of-sight from each level to
    !             all other layers.
    !  x1.........Water vapor path (e-type ) and working dimension
    !  x2.........Water vapor path (band-center) and working dimension
    !  cc.........planck function at level temperature for co2 bands.
    !  x3.........water vapor path (band-wings), working dimension
    !  x4.........Working dimension
    !  ch.........probability of clear line-of-sight from level to top of
    !             the atmosphere.
    !  css........Large scale cloud amount
    !             css=css*(1-exp(-0.01*dp)) for ice cloud t < 253.0
    !  ccu........Cumulus cloud amount
    !=========================================================================!
    ! >>> icld=1     : old cloud emisivity setting                           !
    !       ccu = ccu*(1-exp(-0.05*dp))                                      !
    !       css = css*(1-exp(-0.01*dp))          for ice cloud t<253.0       !
    !       css = css*(1-exp(-0.05*dp))          for     cloud t>253.0       !
    ! >>> icld=2     : new cloud emisivity setting                           !
    !       ccu = 1.0-exp(-0.12*ccu*dp)                                      !
    !       css = 0.0                                    for  t<-82.5c        !
    !       css = 1-exp(-1.5e-6*(t-tcrit)**2*css*dp)     for -82.5<t<-10.0    !
    !       css = 1-exp(-5.2e-3*(t-273.)-0.06)*css*dp)   for -10.0<t< 0.0     !
    !       css = 1-exp(-0.06*css*dp)                    for t> 0.0c          !
    ! >>> icld = 3   : ccm3 based cloud emisivity                             !
    !=========================================================================!
    !==========================================================================

    INTEGER, INTENT(in   ) :: ncols
    INTEGER, INTENT(in   ) :: kmax
    INTEGER, INTENT(in   ) :: nlcs
    REAL(KIND=r8),    INTENT(in   ) :: h1p
    REAL(KIND=r8),    INTENT(inout  ) :: cs  (ncols,kmax+2,nlcs)
    REAL(KIND=r8),    INTENT(inout  ) :: x1  (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: x2  (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: cc  (ncols,kmax+3)
    REAL(KIND=r8),    INTENT(inout  ) :: x3  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: x4  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(inout  ) :: ch  (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(in   ) :: css (ncols,kmax+2)
    REAL(KIND=r8),    INTENT(in   ) :: ccu (ncols,kmax+2)

    INTEGER :: i
    INTEGER :: k
    INTEGER :: j
    INTEGER :: ip
    INTEGER :: il
    INTEGER :: ix
    INTEGER :: ipm1
    INTEGER :: ipp1

    DO k=1,(kmax+2)
       DO i=1,(ncols)
          ch(i,k) = h1p
          cc(i,k) = h1p
          x1(i,k) = h1p
          x2(i,k) = h1p
          x3(i,k) = h1p
       END DO
    END DO

    DO k=1,(kmax+2)
       DO j=1,(kmax+2)
          DO i=1,(ncols)
             cs(i,j,k) = h1p
          END DO
       END DO
    END DO

    DO ip=2,(kmax+2)
       DO i=1,ncols
          x1(i,ip) = h1p - ccu(i,ip-1)
       END DO

       DO i=1,ncols
          x1(i,ip)=MIN(x1(i,ip),x1(i,ip-1))
       END DO
       DO i=1,ncols
          ch(i,ip) = ch(i,ip-1) * (h1p - css(i,ip-1))
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i=1,(ncols)
          ch(i,k+1) = ch(i,k+1) * x1(i,k+1)
       END DO
    END DO

    DO il=1,(kmax+1)
       ip=(kmax+2)-il
       DO i=1,ncols
          x2(i,ip) = h1p - ccu(i,ip)
       END DO

       DO i=1,ncols
          x2(i,ip)=MIN(x2(i,ip),x1(i,ip+1))
       END DO

       DO i=1,ncols
          cc(i,ip) = cc(i,ip+1) * (h1p - css(i,ip))
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i=1,(ncols)
          cc(i,k) = cc(i,k) * x2(i,k)
       END DO
    END DO

    DO ip=2,(kmax+2)
       ipm1   = ip - 1

       DO ix=1,ipm1
          DO i=1,ncols
             x4(i,ix)    = h1p - ccu(i,ip-1)
             cs(i,ix,ip) = cs(i,ix,ip-1) * (h1p - css(i,ip-1))
          END DO
       END DO

       IF (ip .GT. 2) THEN
          DO j=1,ipm1
             DO i=1,ncols
                cs(i,j,ip-1) = cs(i,j,ip-1) * x3(i,j)
             END DO
          END DO
       END IF

       DO k=1,ipm1
          DO i=1,ncols
             x4(i,k)=MIN(x4(i,k),x3(i,k))
             x3(i,k) = x4(i,k)
          END DO
       END DO

    END DO

    DO j=1,ipm1
       DO i=1,ncols
          cs(i,j,(kmax+2)) = cs(i,j,(kmax+2)) * x4(i,j)
       END DO
    END DO

    DO k=1,kmax+2
       DO i=1,ncols
          x3(i,k) = h1p
       END DO
    END DO

    DO il=1,(kmax+1)
       ip   = (kmax+2) - il
       ipp1 = ip + 1

       DO ix=ipp1,(kmax+2)
          DO i=1,ncols
             x4(i,ix)    = h1p - ccu(i,ip)
             cs(i,ix,ip) = cs(i,ix,ip+1) * (h1p - css(i,ip))
          END DO
       END DO

       IF (il .GT. 1) THEN
          DO k=ip,((kmax+1)-ip)! DO k=ip,((kmax+2)-ip)
             DO  i=1, ncols
                cs(i,k+2,ip+1) = cs(i,k+2,ip+1) * x3(i,k+2)
             END DO
          END DO
       END IF

       DO k=ipp1,((kmax+2)-ip)
          DO i=1,ncols
             x4(i,j) = MIN(x4(i,j),x3(i,j))
             x3(i,k) = x4(i,k)
          END DO
       END DO

    END DO

    DO k=ipp1,((kmax+2)-ip)
       DO i=1, ncols
          cs(i,k,ip) = cs(i,k,ip) * x4(i,k)
       END DO
    END DO

  END SUBROUTINE cldslw


  ! lwrad  :compute upward and downward fluxes.
  SUBROUTINE lwrad( &
       ! Model Info and flags
       ncols ,kmax  ,nls   ,nlcs  , noz   ,icld  ,&
       ! Atmospheric fields
       pl20  ,pl    ,tl    ,ql    , o3l   ,tg    ,&
       co2m,                                    &
       ! LW Radiation fields 
       ulwclr,ulwtop,atlclr,atl   ,rsclr , rs    ,&
       dlwclr,dlwbot,                             &
       ! Cloud field and Microphysics
       cld   ,clu   ,clwp  ,fice  ,rei   ,emisd     )
    IMPLICIT NONE
    ! input variables
    !
    !     noz,  pl,  pl20, tl,  tg,  ql,  o3l,  cld,  clu
    !
    !     output variables
    !
    !     atl, atlclr,  rs,  rsclr, ulwtop, ulwclr, dlwbot, dlwclr
    !
    !     parameter list variables
    !
    !     nim......number of grid points around a latitude circle.
    !     nlm......number of model layers.
    !     nlmp1....nlm plus one.
    !     nlmp2....nlm plus two.
    !     nlmp3....nlm plus three.
    !     nls......number of layers in the stratosphere.
    !
    ! local variables
    !
    !     b250.....planck function table for water vapor bands; center and
    !              wing.
    !     ntm1.....number of rows in b250 - 1.
    !     temp1....lowest temperature for which b250 is tabulated.
    !     dtb......temperature increment in b250.
    !     nup1.....number of columns in gl and coeff.
    !     ubar.....scaled water vapor path length in window.
    !     vbar.....scaled water vapor path length in center.
    !     wbar.....scaled water vapor path length in wing.
    !     rawi.....water vapor amount in layer.
    !     ozai.....ozone amount in layer.
    !     shh......planck function at level temperature for water vapor
    !              bands.
    !     shi......g-function for a path from level to top of atmosphere.
    !     txuf.....1.used as matrix of g-functions for paths from each level
    !                to all other layers.
    !              2.used for transmission in co2 band.
    !              3.used for transmission in ozone band.
    !              4.in cldslw used for probability of clear line-of-sight
    !                from each level to all other layers for max overlap.
    !     wv,ww....interpolated value of gl for band center and wing
    !              respectively.
    !     yv,yw....linear term in temperature correction for band center
    !              and wing respectively.
    !     zv,zw....quadratic term in temperature correction for band center
    !              and wing respectively.
    !     blkco2...planck function table for co2 bands.
    !     blkwin...planck function table for ozone band.
    !     cc.......planck function at level temperature for co2 bands.
    !     vbarm....planck function at level temperature for ozone band.
    !     pp.......doppler broadening cut-off.
    !     pscalv...scaled co2 amount in band center.
    !     pscalw...scaled co2 amount in band wing.
    !     dx.......parameterized optical depth of water vapor line in co2
    !              band wing.
    !     dy.......parameterized optical depth of water vapor continuum in
    !              co2 band wing, also used for ozone band wing.
    !     sv,sw....minimum in table of log water vapor amount in band center
    !              and wing respectively.
    !     cs.......probability of clear line-of-sight from each level to
    !              all other layers.
    !     ch.......probability of clear line-of-sight from level to top of
    !              the atmosphere.
    !     cc.......probability of clear line-of-sight from level to surface.
    !     ct.......probability of clear line-of-sight from level to top of
    !              atmosphere for maximun overlap.
    !     cu.......probability of clear line-of-sight from level to surface
    !              for maximum overlap.
    !
    !cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
    ! >>> icld=1     : old cloud emisivity setting                      c
    !       ccu = ccu*(1-exp(-0.05*dp))                                 c
    !       css = css*(1-exp(-0.01*dp))          for ice cloud t<253.0  c
    !       css = css*(1-exp(-0.05*dp))          for     cloud t>253.0  c
    ! >>> icld=2     : new cloud emisivity setting                      c
    !       ccu = 1.0-exp(-0.12*ccu*dp)                                 c
    !       css = 0.0                                for  t<-82.5c      c
    !       css = 1-exp(-1.5e-6*(t-tcrit)**2*css*dp) for -82.5<t<-10.0c c
    !       css = 1-exp(-5.2e-3*(t-273.)-0.06)*css*dp)for -10.0<t< 0.0c c
    !       css = 1-exp(-0.06*css*dp)                 for t> 0.0c       c
    ! >>> icld = 3   : ccm3 based cloud emisivity                       c
    !cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
    !
    ! ncols......Number of grid points on a gaussian latitude circle
    ! kmax......Number of grid points at vertical
    ! nls.......number of layers in the stratosphere.
    ! nlcs......nlcs=30
    ! noz.......constant logical noz = .FALSE.
    ! tg........Surface Temperature (K)
    ! pl20......pl20(i,k)=gps(i)*sigml(kflip) where
    !                         gps   =  surface pressure   (mb)
    !                         sigml =  sigma coordinate at bottom of layer
    ! pl........Pressure at Middle of Layer(mb)
    ! tl........Temperature at middle of Layer (K)
    ! ql........Specific Humidity at middle of layer (g/g)
    ! o3l.......Ozone Mixing ratio at middle of layer (g/g)
    ! cld.......Large scale cloud amount in layers
    ! clu.......Cumulus cloud amount in layers
    ! ulwclr....Upward flux at top in clear case (W/m2)
    ! ulwtop....Upward flux at top (W/m2)
    ! atlclr....Heating rate in clear case (K/s)
    ! atl.......Heating rate (K/s)
    ! rsclr.....Net surface flux in clear case (W/m2 )
    ! rs........Net surface flux
    ! dlwclr....Downward flux at surface in clear case (W/m2 )
    ! dlwbot....Downward flux at surface (W/m2 )
    ! clwp
    ! fice......controle of change of fase of water
    ! rei.......determine rei as function of normalized pressure
    ! emisd.....emis(i,kflip) = 1.- EXP(-1.66*rkabs(i,k)*clwp(i,k))
    ! co2m....co2val is wgne standard value in ppm "co2val = /345.0/
    !

    ! Model Info and flags
    INTEGER, INTENT(in   ) :: ncols
    INTEGER, INTENT(in   ) :: kmax
    INTEGER, INTENT(in   ) :: nls
    INTEGER, INTENT(in   ) :: nlcs
    LOGICAL, INTENT(in   ) :: noz
    INTEGER, INTENT(in   ) :: icld

    ! Atmospheric fields
    REAL(KIND=r8),    INTENT(in   ) :: pl20  (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: pl    (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: tl    (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: ql    (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: o3l   (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: tg    (ncols)
    REAL(KIND=r8),    INTENT(in   ) :: co2m(ncols,kMax)  !mol/mol 
    ! LW Radiation fields 
    REAL(KIND=r8),    INTENT(inout  ) :: ulwclr(ncols)
    REAL(KIND=r8),    INTENT(inout  ) :: ulwtop(ncols)
    REAL(KIND=r8),    INTENT(inout  ) :: atlclr(ncols,kmax)
    REAL(KIND=r8),    INTENT(inout  ) :: atl   (ncols,kmax)
    REAL(KIND=r8),    INTENT(inout  ) :: rsclr (ncols)
    REAL(KIND=r8),    INTENT(inout  ) :: rs    (ncols)
    REAL(KIND=r8),    INTENT(inout  ) :: dlwclr(ncols)
    REAL(KIND=r8),    INTENT(inout  ) :: dlwbot(ncols)

    ! Cloud field and Microphysics
    REAL(KIND=r8),    INTENT(in   ) :: cld   (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: clu   (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: clwp  (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: fice  (ncols,kmax)
    REAL(KIND=r8),    INTENT(in   ) :: rei   (ncols,kmax)
    REAL(KIND=r8),    INTENT(inout) :: emisd (ncols,kmax)

    REAL(KIND=r8)    :: txuf  (ncols,kmax+2,kmax+2)
    REAL(KIND=r8)    :: cs    (ncols,kmax+2,nlcs)
    REAL(KIND=r8)    :: tv1   (ncols,kmax+3)
    REAL(KIND=r8)    :: tv2   (ncols,kmax+3)
    REAL(KIND=r8)    :: tui   (ncols,kmax+3)
    REAL(KIND=r8)    :: x1    (ncols,kmax+3)
    REAL(KIND=r8)    :: x2    (ncols,kmax+3)
    REAL(KIND=r8)    :: cc    (ncols,kmax+3)
    REAL(KIND=r8)    :: rawi  (ncols,kmax+3)
    REAL(KIND=r8)    :: x3    (ncols,kmax+2)
    REAL(KIND=r8)    :: x4    (ncols,kmax+2)
    REAL(KIND=r8)    :: ch    (ncols,kmax+2)
    REAL(KIND=r8)    :: dp    (ncols,kmax+2)
    REAL(KIND=r8)    :: css   (ncols,kmax+2)
    REAL(KIND=r8)    :: ccu   (ncols,kmax+2)
    REAL(KIND=r8)    :: shi   (ncols,kmax+2)
    REAL(KIND=r8)    :: shh   (ncols,kmax+2)
    REAL(KIND=r8)    :: shu   (ncols,kmax+1)
    REAL(KIND=r8)    :: suma  (ncols)
    REAL(KIND=r8)    :: sumsav(ncols)
    LOGICAL :: bitx  (ncols*(kmax+3))


    REAL(KIND=r8) :: pai   (ncols,kmax+2)
    REAL(KIND=r8) :: tai   (ncols,kmax+2)
    REAL(KIND=r8) :: ozai  (ncols,kmax+2)
    REAL(KIND=r8) :: ubar  (ncols,kmax+2)
    REAL(KIND=r8) :: vbar  (ncols,kmax+2)
    REAL(KIND=r8) :: wbar  (ncols,kmax+2)
    REAL(KIND=r8) :: ubarm (ncols,kmax+2)
    REAL(KIND=r8) :: vbarm (ncols,kmax+2)
    REAL(KIND=r8) :: wbarm (ncols,kmax+2)
    REAL(KIND=r8) :: fluxu (ncols,kmax+2)
    REAL(KIND=r8) :: fluxd (ncols,kmax+2)


    REAL(KIND=r8)      :: h0p
    REAL(KIND=r8)      :: h1p
    REAL(KIND=r8)      :: h1p5
    REAL(KIND=r8)      :: hp5
    REAL(KIND=r8)      :: dtb
    REAL(KIND=r8)      :: dtbinv
    REAL(KIND=r8)      :: pr(2)
    INTEGER   :: ntm1
    INTEGER   :: imnpnp
    LOGICAL   :: ozone

    REAL(KIND=r8)      :: emis (ncols,kmax)
    REAL(KIND=r8)      :: emis1(ncols,kmax+2)
    REAL(KIND=r8)      :: rkabs(ncols,kmax)

    REAL(KIND=r8)      :: h1p02
    REAL(KIND=r8)      :: h6p08
    REAL(KIND=r8)      :: tice

    INTEGER  :: ls1
    INTEGER  :: ls2
    INTEGER  :: imls1
    INTEGER  :: imlsm1
    INTEGER  :: imlm1
    INTEGER  :: imt2
    INTEGER  :: npmls1
    INTEGER  :: npmls2
    INTEGER  :: i
    INTEGER  :: ip
    INTEGER  :: ipm1
    INTEGER  :: ix
    INTEGER  :: ipp1
    INTEGER  :: l
    INTEGER  :: k
    INTEGER  :: kflip
    REAL(KIND=r8)     :: fac
    REAL(KIND=r8)     :: h3ppm
    REAL(KIND=r8)     :: tcrit
    REAL(KIND=r8)     :: ecrit
    REAL(KIND=r8)     :: d642
    REAL(KIND=r8)     :: pre   (2)

    h1p02 =   1.02e0_r8
    h6p08 =   6.0811e0_r8
    tice  = 273.16_r8
    !
    !     copy parameters into local variables
    !
    ls1    = nls+1
    ls2    = nls+2
    imls1  = ncols*ls1
    imlsm1 = ncols*(nls-1)
    imnpnp = (ncols*(kmax+2))*(kmax+2)
    imlm1  = ncols*(kmax-1)
    imt2   = ncols*2
    npmls1 = ncols* ((kmax+2) - ls1)
    npmls2 = ncols* ((kmax+2) - ls2)
    !
    !     fac converts to degrees / time step
    !
    fac    = 9.8e-1_r8 / 1.0030e04_r8
    h1p    = 1.0e0_r8
    h0p    = 0.0e0_r8
    h1p5   = 1.5e0_r8
    hp5    = 0.5e0_r8
    h3ppm  = 3.0e-6_r8
    tcrit  = tice - 82.5_r8
    ecrit  = 0.007884375_r8
    ntm1   = 31
    dtb    = 5.0e0_r8
    dtbinv = h1p / dtb
    pre(1) = h1p / 2.75e02_r8
    pre(2) = h1p / 5.5e02_r8
    d642   = h1p / 6.426e02_r8
    pr(1)  = h1p / 3.0e01_r8
    pr(2)  = h1p / 3.0e02_r8
    emis   = 0.0e0_r8
    emis1  = 0.0e0_r8
    emisd  = 0.0e0_r8
    !
    !     set ozone logical variable
    !
    ozone = (.NOT. noz)
    !
    !     ptop and dp at top don't change
    !
    DO i = 1, ncols
       dp (i,1)     = h0p
       x1 (i,1)     = h1p
       dp (i,2)     = h1p
       pai(i,1)     = hp5
    END DO

    DO k=1,((kmax+2) - (nls+1))
       DO i = 1, ncols
          rawi(i,nls+1+k) = ql(i,nls+k-1)
       END DO
    END DO

    IF (nls > 1) THEN
       DO k = 1,(nls-1)
          DO i = 1, ncols
             rawi(i,2+k) = h3ppm
          END DO
       END DO
    END IF

    DO k=1,kmax
       DO i = 1, ncols
          pai(i,k+1) = pl(i,k)
          tai(i,k+2) = tl(i,k)
          x1 (i,k+1) = pl20(i,k)
       END DO
    END DO

    DO i = 1, ncols
       pai(i,(kmax+2)) = pl20(i,kmax)
    END DO

    IF ( ozone ) THEN
       DO k = 1,kmax
          DO i = 1, (ncols)
             ozai(i,k+2) = o3l(i,k)
          END DO
       END DO
    END IF

    DO k = 1,kmax
       DO i = 1, ncols
          dp(i,k+2) = x1(i,k+1) - x1(i,k)
       END DO
    END DO
    !
    !     temperature and humidity interpolations
    !
    DO k = 1,kmax
       DO i = 1, ncols
          rawi(i,k+2)=MAX(0.1e-22_r8,rawi(i,k+2))
       END DO
    END DO

    IF ( ozone ) THEN

       DO k = 1,kmax
          DO i = 1, ncols
             ozai(i,k+2)=MAX(0.1e-9_r8,ozai(i,k+2))
          END DO
       END DO

       DO i = 1, ncols
          ozai(i,1) = h0p
          ozai(i,2) = ozai(i,3) * dp(i,2) * h1p02
       END DO

       DO k = 1,kmax
          DO  i = 1, ncols
             ozai(i,k+2) = ozai(i,k+2) * dp(i,k+2)  * h1p02
          END DO
       END DO

    END IF
    !
    !     do temperature interpolation
    !
    DO i = 1, ncols
       rawi(i,1) = h0p
       rawi(i,2) = h3ppm * dp(i,2) * h1p02
    END DO

    DO k=1,kmax
       DO i = 1, ncols
          rawi(i,k+2) = rawi(i,k+2) * dp(i,k+2) * h1p02
          x1(i,k)   = x1(i,k+1) / pai(i,k+1)
       END DO
    END DO

    x2(1:ncols,1:kmax)=LOG(x1(1:ncols,1:kmax))

    DO k = 1,kmax
       DO i = 1, ncols
          x1(i,k)   = pai(i,k+2) / pai(i,k+1)
       END DO
    END DO

    x4(1:ncols,1:kmax)=LOG(x1(1:ncols,1:kmax))

    DO k=1,(kmax-1)
       DO i = 1, ncols
          tui(i,k+2) = tai(i,k+2)  + x2(i,k) / x4(i,k) &
               * (tai(i,k+3) - tai(i,k+2))
       END DO
    END DO
    !
    !     set surface air temperature as mean of lowest layer t and tg
    !
    DO i = 1, ncols
       tui(i,(kmax+2)) = hp5 * ( tai(i,(kmax+2)) + tg(i) )
       tui(i,(kmax+3)) = tg(i)
       tai(i,1)        = tai(i,3)
       tai(i,2)        = tai(i,3)
    END DO

    DO k=1,2
       DO i = 1, ncols
          tui(i,k)     = tai(i,k)
       END DO
    END DO

    !
    !     compute scaled water vapor amounts
    !

    DO i = 1, ncols
       ubar(i,1) = h0p
       vbar(i,1) = h0p
       wbar(i,1) = h0p
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          x3(i,k) = 0.5e-2_r8 * (tai(i,k+1) - 225.0e0_r8)
       END DO
    END DO

    x2(1:ncols,1:kmax+1)=EXP(x3(1:ncols,1:kmax+1))

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          vbar(i,k+1) = rawi(i,k+1) * x2(i,k) &
               * (pai(i,k) * pre(1))
          x3(i,k)     = 1.6e-2_r8 * (tai(i,k+1) - 256.0e0_r8)
       END DO
    END DO

    x2(1:ncols,1:kmax+1)=EXP(x3(1:ncols,1:kmax+1))

    DO k = 1,(kmax+1)
       DO i = 1, (ncols)
          wbar(i,k+1) = rawi(i,k+1) * x2(i,k) &
               * (pai(i,k) * pre(2))
          x1(i,k)     = 1.8e03_r8    / tai(i,k+1) - h6p08
       END DO
    END DO

    x2(1:ncols,1:kmax+1)=EXP(x1(1:ncols,1:kmax+1))

    DO k = 1, (kmax+1)
       DO i = 1, (ncols)
          x2(i,k) = x2(i,k) * pai(i,k)  * d642
          x2(i,k) = x2(i,k) * rawi(i,k+1) &
               * rawi(i,k+1) / dp(i,k+1)
       END DO
    END DO

    DO ip = 2, (kmax+2)
       DO i = 1, ncols
          ubar(i,ip) = ubar(i,ip-1) + x2(i,ip-1)
          vbar(i,ip) = vbar(i,ip-1) + vbar(i,ip)
          wbar(i,ip) = wbar(i,ip-1) + wbar(i,ip)
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          ubarm(i,k+1) = (ubar(i,k+1) + ubar(i,k)) * hp5
          vbarm(i,k+1) = (vbar(i,k+1) + vbar(i,k)) * hp5
          wbarm(i,k+1) = (wbar(i,k+1) + wbar(i,k)) * hp5
       END DO
    END DO

    CALL lwflux(pai   ,tai   ,ozai  ,ubar  ,vbar  ,wbar  ,ubarm ,vbarm , &
         wbarm ,fluxu ,fluxd ,txuf  ,tv1   ,tv2   ,tui   ,x1    , &
         x2    ,cc    ,rawi  ,x3    ,x4    ,ch    ,dp    ,css   , &
         ccu   ,shi   ,shh   ,shu   ,sumsav,h0p   ,h1p   ,h1p5  , &
         hp5   ,dtb   ,dtbinv,pr    ,ntm1  ,ozone ,co2m, &
         ncols ,kmax  )
    !
    !     compute clear sky fluxes only for cloud radiative forcing
    !     zero out downward flux accumulators at l=1
    !
    DO i = 1, ncols
       fluxd(i,1) = h0p
       !
       !     upward flux at the surface
       !
       fluxu(i,(kmax+2)) = sumsav(i)
    END DO
    !
    !     compute downward fluxes
    !
    DO ip = 2, (kmax+2)
       DO i = 1, ncols
          suma(i) = h0p
       END DO
       ipm1 = ip - 1
       DO ix = 1, ipm1
          DO i = 1, ncols
             suma(i) = suma(i) + txuf(i,ix,ip)
          END DO
       END DO
       DO i = 1, ncols
          fluxd(i,ip) = suma(i) - shi(i,ip) + shh(i,ip)
       END DO
    END DO
    !
    !     compute upward fluxes
    !
    DO ip = 1, (kmax+1)
       ipp1      = ip + 1
       DO i = 1, ncols
          suma(i) = h0p
       END DO
       DO ix = ipp1, (kmax+2)
          DO i = 1, ncols
             suma(i) = suma(i) + txuf(i,ix,ip)
          END DO
       END DO
       DO i = 1, ncols
          fluxu(i,ip) = suma(i)  + shu(i,ip) + shh(i,ip)
       END DO
    END DO

    DO k=1,(kmax+2)
       DO  i = 1, (ncols)
          x1(i,k) = fluxd(i,k) - fluxu(i,k)
       END DO
    END DO

    DO i = 1, ncols
       ulwclr(i)  = -x1(i,2)
       rsclr (i)  = -x1(i,(kmax+2))
       dlwclr(i)  = fluxd(i,(kmax+2))
    END DO

    DO k = 1,kmax
       DO i = 1, (ncols)
          atlclr(i,k) = (x1(i,k+1) - x1(i,k+2)) * fac / dp(i,k+2)
       END DO
    END DO

    !
    !     we don't allow clouds between gcm levels in the stratosphere
    !

    DO k=1,kmax
       DO i = 1, (ncols)
          css(i,k+1)   = cld(i,k)
          ccu(i,k+1)   = clu(i,k)
       END DO
    END DO

    DO k=1,(nls+1)
       DO i = 1, ncols
          css(i,k) = h0p
          ccu(i,k) = h0p
       END DO
    END DO
    !
    !     icld = 1     : old cloud emissivity setting
    !     icld = 2     : new cloud emissivity setting
    !
    IF (icld == 1) THEN

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1, ncols
             tv1(i,nls+k+1) = -0.05e0_r8 *  dp(i,nls+k+2)
          END DO
       END DO

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,ncols
             tv2(i,nls+k+1)=EXP(tv1(i,nls+k+1))
             ccu(i,nls+k+1) = ccu(i,nls+k+1) * ( h1p - tv2(i,nls+k+1) )
          END DO
       END DO

       DO l = ls2, (kmax+1)
          DO i = 1, ncols
             !
             !     bash down cloud emissivities
             !
             IF (tl(i,l-1) < 253.0_r8) THEN
                tv1(i,l) = -0.01e0_r8 * dp(i,l+1)
             END IF
          END DO
       END DO

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,  ncols
             tv2(i,nls+k+1)=EXP(tv1(i,nls+k+1))
             css(i,nls+k+1) = css(i,nls+k+1) * ( h1p - tv2(i,nls+k+1) )
          END DO
       END DO

    ELSE IF (icld == 2) THEN

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,  ncols
             tv1(i,nls+k+1) = -0.12e0_r8 * ccu(i,nls+k+1) * dp(i,nls+k+2)
          END DO
       END DO

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,  ncols
             tv2(i,nls+k+1)=EXP(tv1(i,nls+k+1))
             ccu(i,nls+k+1) =  h1p - tv2(i,nls+k+1)
          END DO
       END DO

       DO l = ls2, (kmax+1)
          DO i = 1, ncols
             tv2(i,l) =  tl(i,l-1) - tcrit
          END DO
       END DO

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,  ncols
             tv2(i,nls+k+1) = MAX(tv2(i,nls+k+1),h0p)
             tv1(i,nls+k+1) = MIN(1.5e-6_r8 * tv2(i,nls+k+1) * tv2(i,nls+k+1),ecrit)
          END DO
       END DO

       DO l = ls2, (kmax+1)
          DO i = 1, ncols
             tv2(i,l) =  tl(i,l-1) - tice
          END DO
       END DO

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,  ncols
             tv2(i,nls+k+1) = MIN(MAX(5.2115625e-3_r8 * tv2(i,nls+k+1) + 0.06e0_r8 ,ecrit),0.06_r8)
             bitx(k) = (tv1(i,k).eq.ecrit)
             IF (bitx(k)) tv1(i,k)=tv2(i,k)
          END DO
       END DO

       DO l = ls2, (kmax+1)
          DO i = 1, ncols
             tv1(i,l) =  -tv1(i,l) * css(i,l) * dp(i,l+1)
          END DO
       END DO

       DO k=1,((kmax+2) - (nls+2))
          DO i = 1,  ncols
             tv2(i,nls+k+1)=EXP(tv1(i,nls+k+1))
             css(i,nls+k+1) =  h1p - tv2(i,nls+k+1)
          END DO
       END DO

    ELSE IF ((icld == 3).OR.(icld == 4).OR.(icld == 5).OR.(icld == 6) .OR.(icld == 7)) THEN

       DO k = 1, kmax
          kflip=kmax+1-k
          DO i = 1, ncols
             rkabs(i,k) = 0.090361_r8*(1.0_r8-fice(i,k)) + &
                  (0.005_r8 + 1.0_r8/rei(i,k))*fice(i,k)
             emis(i,k) =MIN(MAX(1.0_r8- EXP(-1.66_r8*rkabs(i,k)*clwp(i,k)),0.0_r8),1.0_r8)
             emisd(i,k)=MIN(MAX(emis(i,kflip),0.0_r8),1.0_r8)
          END DO
       END DO

       DO k=1,kmax
          DO i = 1, (ncols)
             emis1(i,k+1) = emis(i,k)
          END DO
       END DO

       DO k=1,nls+1
          DO i = 1,  ncols
             emis1(i,k) = h0p
          END DO
       END DO

       DO k=nls+2,kmax+1
          DO i = 1, ncols
             css(i,k) = css(i,k) * emis1(i,k)
             ccu(i,k) = ccu(i,k) * emis1(i,k)
          END DO
       END DO

    END IF
    !
    !     get the contribution from cloudiness
    !
    CALL cldslw(ncols ,kmax  ,nlcs  ,h1p   ,cs    ,x1    ,x2    , &
         cc    ,x3    ,x4    ,ch    ,css   ,ccu   )

    DO k=1,(kmax+2)
       DO l=1,(kmax+2)
          DO i = 1, (ncols)
             txuf(i,l,k) = txuf(i,l,k) * cs(i,l,k)
          END DO
       END DO
    END DO

    DO k=1,(kmax+2)
       DO i = 1, (ncols)
          shi(i,k) = shi(i,k) * ch(i,k)
       END DO
    END DO

    DO k=1,(kmax+1)
       DO i = 1, (ncols)
          shu(i,k) = shu(i,k) * cc(i,k)
          shu(i,k) = shu(i,k) + shh(i,k)
       END DO
    END DO
    !
    !     zero out downward flux accumulators at l=1
    !
    DO i = 1, ncols
       fluxd(i,1) = h0p
    END DO
    !
    !     upward flux at the surface
    !
    DO i = 1, ncols
       fluxu(i,(kmax+2)) = sumsav(i)
    END DO
    !
    !     upward flux at the surface was computed before call to cldslw
    !     compute downward fluxes
    !
    DO ip = 2, (kmax+2)

       DO i = 1, ncols
          suma(i) = h0p
       END DO

       ipm1 = ip - 1

       DO ix = 1, ipm1
          DO i = 1, ncols
             suma(i) = suma(i) + txuf(i,ix,ip)
          END DO
       END DO

       DO i = 1, ncols
          fluxd(i,ip) = suma(i) - shi(i,ip) + shh(i,ip)
       END DO

    END DO
    !
    !     compute upward fluxes
    !
    DO ip = 1, (kmax+1)
       ipp1 = ip + 1

       DO i = 1, ncols
          suma(i) = h0p
       END DO

       DO ix = ipp1, (kmax+2)
          DO i = 1, ncols
             suma(i) = suma(i) + txuf(i,ix,ip)
          END DO
       END DO

       DO i = 1, ncols
          fluxu(i,ip) = suma(i) + shu(i,ip)
       END DO

    END DO

    DO k=1,(kmax+2)
       DO i = 1, (ncols)
          x1(i,k) = fluxd(i,k) - fluxu(i,k)
       END DO
    END DO

    DO i = 1, ncols
       ulwtop(i) = -x1(i,2)
       rs(i)     = -x1(i,(kmax+2))
       dlwbot(i) = fluxd(i,(kmax+2))
    END DO

    DO k=1,kmax
       DO i = 1, (ncols)
          atl(i,k) = (x1(i,k+1) - x1(i,k+2)) * fac / dp(i,k+2)
       END DO
    END DO

  END SUBROUTINE lwrad

END MODULE Rad_COLA
